/*** Artikelstamm und Artikelstammdateien ***/

--ICs
CREATE TABLE intcod (
  ic_n                      SMALLINT NOT NULL PRIMARY KEY,
  ic_b                      INTEGER -- Textnummer
 );
--

--Mengeneinheiten
CREATE TABLE mgcode (
    me_cod      integer NOT NULL PRIMARY KEY,
    me_iso      varchar(10),
    -- Mengeneinheit im Standard der UN/ECE
    me_unece    varchar(3),
    me_bez      varchar(50),
    -- Erste Dimension von ME, m² => m, m³ => m, mm² => mm
    me_dim1     smallint,
    me_dimnum   smallint
 );
 --

 --
 CREATE OR REPLACE FUNCTION mgcode__a_iud() RETURNS TRIGGER AS $$
  BEGIN
   IF NOT tg_op='INSERT' THEN
       DELETE FROM mgcodelang WHERE mel_me_cod=old.me_cod AND mel_spr_key=prodat_languages.curr_lang();
   END IF;
   IF NOT tg_op='DELETE' THEN
       INSERT INTO mgcodelang (mel_me_cod, mel_spr_key, mel_iso, mel_txt) VALUES (new.me_cod, prodat_languages.curr_lang(), new.me_iso, new.me_bez);
   END IF;
   RETURN new;
  END $$ LANGUAGE plpgsql;

   CREATE TRIGGER mgcode__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    ON mgcode
    FOR EACH ROW
    EXECUTE PROCEDURE mgcode__a_iud();
--
-- Mengeneinheiten Umrechnung
CREATE TABLE meconvert(
  mc_cod1   INTEGER NOT NULL REFERENCES mgcode ON UPDATE CASCADE ON DELETE CASCADE,
  mc_cod2   INTEGER NOT NULL REFERENCES mgcode ON UPDATE CASCADE ON DELETE CASCADE,
  mc_uf NUMERIC NOT NULL
 );
 --
 CREATE UNIQUE INDEX meconvert_mc_cod1_mc_cod2 ON meconvert (mc_cod1, mc_cod2);
-- verschiedene Fremdsprachen für ME
CREATE TABLE mgcodelang (
  mel_id        SERIAL NOT NULL PRIMARY KEY,
  mel_me_cod        INTEGER NOT NULL REFERENCES mgcode ON UPDATE CASCADE ON DELETE CASCADE,
  mel_spr_key       VARCHAR(5) NOT NULL CONSTRAINT xtt4089 REFERENCES sprach,
  mel_iso       VARCHAR(10), --ISO Übersetzung
  mel_txt       VARCHAR(50)
 );
 --

 CREATE UNIQUE INDEX mgcode_lang_spr ON mgcodelang (mel_me_cod, mel_spr_key);

--

-- Artikelcodes
CREATE TABLE artcod (
  ac_n                  VARCHAR(9) NOT NULL PRIMARY KEY,
  ac_i                  INTEGER NOT NULL REFERENCES intcod ON DELETE RESTRICT,
  ac_b                  VARCHAR(50),
  ac_prkl               INTEGER NOT NULL DEFAULT 1,
  ac_vkpfaktor          NUMERIC NOT NULL DEFAULT 1,
  ac_vkprund            NUMERIC NOT NULL DEFAULT 0.01,
  ac_konto              VARCHAR(30), -- Aufwandskonto
  ac_konto_erl          VARCHAR(12),  -- Erlöskonto #7691
  ac_ks                 VARCHAR(9),
  ac_checknewart        BOOL NOT NULL DEFAULT false,
  ac_txt                TEXT,
  ac_charge_abk         BOOLEAN NOT NULL DEFAULT false,
  ac_charge_wen         BOOLEAN NOT NULL DEFAULT false,
  ac_charge_dat         BOOLEAN NOT NULL DEFAULT false,
  ac_me_code            INTEGER REFERENCES mgcode,
  ac_me_code_stange3m   BOOLEAN NOT NULL DEFAULT false, -- legt bei m zusätzlich die stange 3m an
  ac_auftgi_pre_float   INTEGER NOT NULL DEFAULT 0,     -- Pufferzeit (in Werktagen) für Material (Bedarfstermin) vor Beginn der Fertigung, siehe #7416
  ac_sn_vorgabe         boolean NOT null DEFAULT false  -- Werden Artikel dieser Gruppe beim Wareneingang generierte Seriennummern zugewiesen?
);

CREATE OR REPLACE FUNCTION artcod__b_iu__ac_i__standardme_stk() RETURNS TRIGGER AS $$
  BEGIN
    -- Fertigungsartikel: Standard ME = Stk
    IF     new.ac_i <> 3
       AND new.ac_me_code IS NULL
    THEN
       new.ac_me_code := 1;
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artcod__b_iu__ac_i__standardme_stk
    BEFORE INSERT OR UPDATE
    OF ac_i
    ON artcod
    FOR EACH ROW
    EXECUTE PROCEDURE artcod__b_iu__ac_i__standardme_stk();


--
CREATE OR REPLACE FUNCTION artcod__a_iu() RETURNS TRIGGER AS $$
  BEGIN
    INSERT INTO artcodlang
      (acl_ac_n, acl_spr_key,                     acl_txt)
    VALUES
      (new.ac_n, prodat_languages.current_lang(), new.ac_b)
    ON CONFLICT(acl_ac_n, acl_spr_key) -- INDEX xtt5109
    DO UPDATE SET
      acl_txt = new.ac_b
    WHERE artcodlang.acl_txt IS DISTINCT FROM new.ac_b
    ;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artcod__a_iu
    AFTER INSERT OR UPDATE
    ON artcod
    FOR EACH ROW
    EXECUTE PROCEDURE artcod__a_iu();
--

-- Fremdsprachen Artikelcodes
CREATE TABLE artcodlang (
  acl_id        SERIAL NOT NULL PRIMARY KEY,
  acl_ac_n      VARCHAR(9) NOT NULL CONSTRAINT artcodlang_artcod REFERENCES artcod ON UPDATE CASCADE ON DELETE CASCADE,
  acl_spr_key   VARCHAR(5) NOT NULL CONSTRAINT xtt4089 REFERENCES sprach,
  acl_txt       VARCHAR(50)
);

-- Indizes
    CREATE UNIQUE INDEX xtt5109 ON artcodlang (acl_ac_n, acl_spr_key);
--

-- ****** Vorgabetabellen für Artikelstammdaten : Bezeichnung, Material, Dimension usw. *****
 -- art.ak_bez
 CREATE TABLE art_abez(
    a_bez VARCHAR(50) NOT NULL PRIMARY KEY
 );

 -- art.ak_dim
 CREATE TABLE art_rdim (
    r_krz VARCHAR(20),
    r_dim VARCHAR(50) NOT NULL PRIMARY KEY
 );

 -- art.ak_mat
 CREATE TABLE art_rmatbez(
    r_krz     VARCHAR(20),
    r_bez     VARCHAR(50) NOT NULL PRIMARY KEY,
    r_dichte  NUMERIC(12,6)
 );

 -- art.ak_norm
 CREATE TABLE art_norm(
    a_norm  VARCHAR(50) NOT NULL PRIMARY KEY
 );

 -- Suchbegriffe  vorgaben standardsuchbegriffe Vordefinierte
 -- "art.ak_such"
 CREATE TABLE art_search (
    aks_terms    VARCHAR(75)
 );
----


-- **************************   Artikelstamm  *****************************
CREATE TABLE art (
  ak_ac                 VARCHAR(9) NOT NULL REFERENCES artcod ON UPDATE CASCADE, -- Artikelcode
  ak_fertigung          BOOL NOT NULL DEFAULT FALSE, -- Artikel ist Fertigungsartikel
  ak_nr                 VARCHAR(40) NOT NULL PRIMARY KEY CONSTRAINT xtt16537__ak_nr CHECK (strpos(ak_nr, '%') = 0), -- Artikelnummer
  ak_bez                VARCHAR(100) NOT NULL, -- Bezeichnung
  ak_mat                VARCHAR(50), -- Material
  ak_dim                VARCHAR(50), -- Dimension, Abmessung
  ak_din                VARCHAR(50), -- Materialnorm
  ak_znr                VARCHAR(50), -- Zeichnungsnummer
  ak_idx                VARCHAR(30), -- Revision
  ak_standard_mgc       SMALLINT NOT NULL CONSTRAINT xtt4066__ak_standard_mgc REFERENCES mgcode, -- Mengeneinheit (Standard)
  ak_bfr                INTEGER NOT NULL DEFAULT TSystem.Settings__GetInteger('EK_Lieferfrist_Default', 1) CONSTRAINT xtt4068__ak_bfr CHECK (ak_bfr>=0), -- Beschaffungsfrist in Tagen. Einkaufseitig als 'Arbeitstage' also ohne WE und FT betrachtet.
  --Felderweiterungen #6507
  ak_rust_et            NUMERIC(12,4),
  ak_fertk_et           NUMERIC(12,4),
  ak_matk_et            NUMERIC(12,4),
  ak_awkost_et          NUMERIC(12,4),
  ak_hest_et            NUMERIC(12,4) NOT NULL DEFAULT 0,
  ak_vkpbas_et          NUMERIC(12,4) NOT NULL DEFAULT 0,
  --
  ak_hest               NUMERIC(12,4) NOT NULL DEFAULT 0, -- Selbstkosten (reine Herstellkosten)
  ak_rust               NUMERIC(12,4), -- Rüstkosten
  ak_fertk              NUMERIC(12,4), -- Montagekosten
  ak_matk               NUMERIC(12,4), -- Materialkosten
  ak_awkost             NUMERIC(12,4), -- Auswärtskosten
  ak_los                NUMERIC(12,2) DEFAULT 1, -- Losgröße
  ak_vkpbas             NUMERIC(12,4) NOT NULL DEFAULT 0, -- Verkaufspreis
  --ak_vkpbrutto        BOOL NOT NULL DEFAULT FALSE,
  ak_vkpfaktor          NUMERIC(12,2), -- Verkaufspreisfaktor
  ak_vkprund            NUMERIC(12,2) NOT NULL DEFAULT 0.01, -- Rundungsgenauigkeit
  ak_vkpbasfix          NUMERIC(12,4), -- Festpreis
  ak_lagdim             SMALLINT,
  ak_pknr               SMALLINT NOT NULL DEFAULT 1 CONSTRAINT xtt4067__ak_pknr CHECK (ak_pknr>0), --Preisklasse
  ak_prognr             SMALLINT, -- CNC/NC-Program
  ak_zeko               VARCHAR(12), --Kontierung Buchhaltung, Erloeskonto (Vorschlag)
  ak_awko               VARCHAR(12), --Kontierung Buchhaltung, Aufwandskonto
  ak_min                NUMERIC(12,4) NOT NULL DEFAULT 0, -- Mindestbestand: d.i. Sicherheitsbestand ist nie verfügbar, Unterschreitung löst Bedarf aus
  ak_melde              NUMERIC(12,4),                    -- Meldebestand: "Meldebedarf" an Einkauf, siehe Ass. WaWi Meldebestand
  ak_soll               NUMERIC(12,4),                    -- Sollbestand: Menge bis zu der bestellt werden soll, siehe Bestellvorschläge
  ak_soll_is_max        BOOLEAN NOT NULL DEFAULT false,   -- Sollbestand ist gleich der Maximalbestand. Darf nicht überschritten werden (insb. durch Losrundung und Bestellvorschlag), siehe #13214
  ak_tot                NUMERIC(12,4) NOT NULL DEFAULT 0, -- Lagernd, total
  ak_res                NUMERIC(12,4) NOT NULL DEFAULT 0, -- Verkauft, reserviert
  ak_bes                NUMERIC(12,4) NOT NULL DEFAULT 0, -- Bestellt, eingekauft
  ak_verfueg            NUMERIC(12,4), -- verfügbare Menge: Lagernd + Bestellt - Verkauft - Mindestbestand
  ak_lag                BOOLEAN NOT NULL DEFAULT TRUE, -- Lagerartikel oder nicht
  ak_stat               VARCHAR(3), --Beistellung usw.usf. siehe auch st_stat
  ak_zolp1              NUMERIC(12,2), -- Kundenspezifische Felder numerisch 1..3
  ak_zolp2              NUMERIC(12,2),
  ak_zolp3              NUMERIC(12,2),
  ak_zolp1b             VARCHAR(75), -- Kundenspezifische Felder alphanumerisch 1..3
  ak_zolp2b             VARCHAR(75),
  ak_zolp3b             VARCHAR(75),
  ak_allg1              VARCHAR(100), -- Kundenspezifische Information
  ak_ispruefmittel      BOOLEAN NOT NULL DEFAULT false, -- Artikel ist ein Prüfmittel. Automatisch gesetzt wenn als PM aufgenommen. Wenn als Werkzeug inventarisiert, muß diese Inventarnummer gleichzeitig prüfmittel sein. http://redmine.prodat-sql.de/issues/898
  ak_allgb2             BOOLEAN, -- Kundenspezifisches Flag
  ak_ordk               VARCHAR(50), -- Ordnerkennung
  ak_txt                TEXT, -- Zusatztext
  ak_auftxt             TEXT, -- Vorgabetexte: Angebot/Auftrag
  ak_auftxt_rtf         TEXT,
  ak_faktxt             TEXT, -- Faktura
  ak_faktxt_rtf         TEXT,
  ak_lieftxt            TEXT, -- Lieferschein
  ak_lieftxt_rtf        TEXT,
  ak_bestxt             TEXT, -- Bestellung
  ak_bestxt_rtf         TEXT,
  ak_gewicht            NUMERIC(12,4), -- Gewicht in kg
  ak_hersteller         VARCHAR(21), --X TableContraints: REFERENCES adk ON UPDATE CASCADE -- Hersteller
  ak_herstelleraknr     VARCHAR(40), -- Artikelnummer des Herstellers
  ak_herstelleraknr_url VARCHAR,     -- URL der Produktseite
  ak_nrhl               VARCHAR(40),
  ak_auslauf            DATE, -- Ablaufdatum, Auslaufdatum
  ak_auslauf_aufbrauch  BOOLEAN NOT NULL DEFAULT FALSE,  -- Artikel nach aufbrauchen ungültig, Auslaufartikel
  ak_ersatzaknr         VARCHAR(40), -- Ersatzartikel bei Auslaufartikel
  ak_norm               VARCHAR(50), -- D ask.sql: REFERENCES qsnorm(qs_ident) ON UPDATE CASCADE -- Fertigungsnorm
  ak_canRabatt          BOOLEAN NOT NULL DEFAULT TRUE, -- Artikel ist rabattfähig
  --ak_robomat          VARCHAR(50),    -- Materialnummer für Verschachteln
  ak_robolos            NUMERIC(12,2),  -- Teile pro Strang
  ak_robodim            VARCHAR(50),    -- Blechtafel Dimension
  ak_robolzstr          NUMERIC(12,2),
  ak_neuanlage          BOOLEAN NOT NULL DEFAULT FALSE,
  ak_ks                 VARCHAR(9),                   -- Kostenstelle
  ak_sernrreq           BOOL NOT NULL DEFAULT FALSE,  -- Seriennummernpflicht > (lgs_psernr) lgs_sernr agmi_sernr (agplog_agmi_sernr_old agplog_agmi_sernr_new il_SerNrString) wr_sernr
  ak_chnrreq            BOOL NOT NULL DEFAULT FALSE,  -- Chargennummernpflicht > w_lgchnr lg_chnr l_lgchnr (lo_lgchnr_old lo_lgchnr_new)
  ak_konsreq            BOOL NOT NULL DEFAULT FALSE,  -- Konservierungsdatumpflicht > w_konsDat w_konsEnde lg_konsDat lg_konsEnde
  ak_slort              VARCHAR(75), -- Standardlagerort
  ak_slbereich          VARCHAR(50), -- Lagerbereich zum Standardlagerort ak_slort aus lgb_name #6259
  ak_such               VARCHAR(75), -- Suchbegriffe
  ak_ean                VARCHAR(20), -- European Article Number
  ak_gtin               VARCHAR(20), -- Global Trade Item Number
  ak_warencode          VARCHAR(20), -- Zollwarencode TARIC etc.
  ak_kunde              VARCHAR(21), -- X TableContraints: REFERENCES adk ON UPDATE CASCADE
  ak_l_iso              VARCHAR(5),  -- X TableContraints: REFERENCES laender ON UPDATE CASCADE -- Ursprungsland
  ak_lerkl_ad_krz       VARCHAR(21), -- X TableContraints: REFERENCES adk ON UPDATE CASCADE -- 'Stammlieferantenerklärung von' Lieferant der Angabe des Herkunftslandes
  ak_lerkldat           DATE,        -- 'Stammlieferantenerklärung vom' Datum der Angabe des Herkunftslandes
  ak_lerkldatbis        DATE,        -- Ablaufdatum der Ursprungserklärung  Ticket 7298 PHKO
  ak_sperrlag           BOOLEAN DEFAULT FALSE, -- Status "Automatische Lagerort-Sperre", so dass der Lagerort dann erst durch Prüfer frei gegeben werden muss
  ak_gefahrgut          BOOLEAN DEFAULT FALSE, -- Status Gefahrgut für Anzeige in Gefahrstoffkataster und Folgeprozesse
  CONSTRAINT xtt25412 CHECK(ak_lerkldat<=ak_lerkldatbis) -- #7951
 );
 --
 -- Indizes in \0400 Indicies\B art.sql

 --
  CREATE TRIGGER art_delete_abk_project_structure
   AFTER DELETE
   ON art
   FOR EACH ROW
   EXECUTE PROCEDURE table_delete_abkstru();
 --

 --
 CREATE OR REPLACE FUNCTION art__a_iu_keywordsearch() RETURNS TRIGGER AS $$
  BEGIN
   PERFORM TSystem.kws_create_keywords(new.*);
   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__a_iu_keywordsearch
   AFTER INSERT OR UPDATE
   OF ak_ac, ak_nr, ak_bez, ak_znr, ak_mat, ak_din, ak_dim, ak_such, ak_ordk, ak_allg1
   ON art
   FOR EACH ROW
   EXECUTE PROCEDURE art__a_iu_keywordsearch();


 -- #7069 Anlegen von Parametern je nach AC
  CREATE TRIGGER art__a_iu__create_autoparams
   AFTER INSERT OR UPDATE
   OF ak_ac, ak_gefahrgut
   ON art
   FOR EACH ROW
   EXECUTE PROCEDURE TRecnoParam.CreateAutoParams();
 --

 --
 CREATE OR REPLACE FUNCTION art__b_05_iu__ak_nr() RETURNS TRIGGER AS $$
  BEGIN
    -- Artikelnummer immer groß schreiben.
    -- Artikel mit Leerzeichen am Anfang und Ende komplett verhindern, siehe #13783.
    new.ak_nr := trim(upper(new.ak_nr));

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__b_05_iu__ak_nr
    BEFORE INSERT OR UPDATE
    OF ak_nr
    ON art
    FOR EACH ROW
    EXECUTE PROCEDURE art__b_05_iu__ak_nr();
 --

 --
 CREATE OR REPLACE FUNCTION art__b_10_iu() RETURNS TRIGGER AS $$
  DECLARE
      _ic integer;
  BEGIN

    -- Artikel mit IC 50 sind nicht lagergeführt
    _ic := ic_for_ac( new.ak_ac );
    IF _ic = 50 THEN
        new.ak_lag := false;
    END IF;
    --
    IF new.ak_gewicht = 0 THEN
        new.ak_gewicht := null;
    END IF;
    --
    IF new.ak_sernrreq
       AND TSystem.Settings__GetBool('art.ak_sernrreq__forces__ak_chnrreq', true) -- Seriennummernpflicht bedeutet Chargennummernpflicht. Experimental (KTA). Zusammenlegen von Lagerorten wenn diese Seriennummern haben, wird knallen. Bei KTA/HPCHO werde nur Seriennummern beim Lagerabgang gescannt. Voher gibt es keine SerNr
    THEN
        -- Chargennumern sind Pflicht, wenn Seriennummern eingegeben werden müssen http://redmine.prodat-sql.de/issues/6252
        new.ak_chnrreq := true;
    END IF;
    --
    new.ak_fertigung := coalesce( _ic = 10, new.ak_fertigung ) OR new.ak_fertigung;
    --
    IF tg_op = 'INSERT' THEN
        IF new.ak_standard_mgc IS null THEN

            -- StandardME des Artikelcodes
            new.ak_standard_mgc := ac_me_code FROM artcod WHERE ac_n = new.ak_ac;
        END IF;
    END IF;
    --
    IF new.ak_bfr IS null THEN

        --- #11261  new.ak_bfr:=0;
        new.ak_bfr := TSystem.Settings__GetInteger( 'EK_Lieferfrist_Default', 1 );
    END IF;

    --Verfügbarkeit
        -- Lagernd + Bestellt - Verkauft - Mindestbestand
        -- Sperrlager (die verfügbar sind) nicht mehr abziehen aufgrund von
        --   ausschließlicher Berücksichtigung der Verfügbarkeit (ist schon in ak_tot), siehe #7381.
    new.ak_verfueg := new.ak_tot + new.ak_bes - new.ak_res - new.ak_min;
    --Hänel-Lift
    IF TSystem.Settings__Get( 'KUNDE' ) = 'LOLL' THEN

        --wir ersetzen alle * durch S bei Hänel
        new.ak_nrhl := REPLACE( new.ak_nr, '*', 'S' );
    ELSE

        --wir ersetzen alle * durch X bei Hänel
        new.ak_nrhl := REPLACE( new.ak_nr, '*', 'X' );
    END IF;

    --Neu angelegte Artikel markieren
    IF ( TG_OP = 'INSERT' ) THEN
        IF ac_checknewart FROM artcod WHERE new.ak_ac = ac_n THEN
            new.ak_neuanlage := true;
        ELSE
            new.ak_neuanlage := false;
        END IF;
        --
        IF    new.ak_kunde IS null
          AND _ic IN ( 10,50 ) THEN

            --gibt '' zurück, falls nicht gefunden
            new.ak_kunde := TSystem.Settings__Get( 'ak_kunde' );
            IF new.ak_kunde = '' THEN
                new.ak_kunde := null;
            END IF;
        END IF;
    END IF;
    --

    --HÄNEL (auch Durchmesser)
    new.ak_nrhl := REPLACE( new.ak_nrhl, 'Ø', 'D' );
    new.ak_nrhl := REPLACE( new.ak_nrhl, chr( 216 ), 'D' );
    new.ak_nrhl := REPLACE( new.ak_nrhl, '''', 'H' );
    new.ak_nrhl := REPLACE( new.ak_nrhl, '#', 'K' );
    new.ak_nrhl := REPLACE( new.ak_nrhl, '&', 'U' );


    --Meldebestand ist immer mindestens Mindestbestand
    IF    coalesce( new.ak_min,0 ) > 0
      AND coalesce( new.ak_melde, 0 ) < coalesce( new.ak_min, 0 ) THEN
        new.ak_melde := new.ak_min;
    END IF;
    --
    IF    new.ak_melde IS NOT null
      AND coalesce( new.ak_soll, 0 ) < new.ak_melde THEN
        new.ak_soll := new.ak_melde;
    END IF;

    -- Wenn kein Soll-Bestand angegeben ist, kann Soll-Bestand nicht Maximalbestand sein.
    IF    new.ak_soll_is_max
      AND coalesce( new.ak_soll, 0 ) <= 0 THEN
        new.ak_soll_is_max := false;
    END IF;

    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__b_10_iu
    BEFORE INSERT OR UPDATE
    ON art
    FOR EACH ROW
    EXECUTE PROCEDURE art__b_10_iu();
 --
 -- #9033 Reset der QS-Freigabe bei neuem Zeichnungsindex oder Änderung der Artikelnummer
 CREATE OR REPLACE FUNCTION art__a_u__reset_qs_freigabe() RETURNS TRIGGER AS $$
  BEGIN
   UPDATE opl SET op_qs_freigabe = NULL, op_stat = NULL WHERE op_n = new.ak_nr;
   IF FOUND THEN -- #13045
     PERFORM PRODAT_MESSAGE(lang_text(28267), 'Information'); -- #10822
   END IF;

   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__a_u__reset_qs_freigabe
   AFTER UPDATE
   OF ak_nr, ak_znr, ak_idx
   ON art
   FOR EACH ROW
   EXECUTE PROCEDURE art__a_u__reset_qs_freigabe() ;
 --

 --

 -- wenn eine neue StandardMengeneinheit und diese noch nicht da in artmgc dann gleich mit anlegen
 CREATE OR REPLACE FUNCTION art__a_iu() RETURNS TRIGGER AS $$
  BEGIN
    IF current_user = 'syncro' THEN
        RETURN new;
    END IF;

    IF TG_OP = 'UPDATE' THEN

        IF new.ak_hest <> old.ak_hest THEN
            PERFORM TArtikel.ain_hest_herkunft__set( new.ak_nr );
        END IF;

        IF
               ( old.ak_tot <> new.ak_tot )
            OR ( old.ak_res <> new.ak_res )
            OR ( old.ak_bes <> new.ak_bes )
            OR ( old.ak_min <> new.ak_min )
        THEN
            RETURN new; --das ist eine interne Änderung (Lagerzugang usw)
        END IF;

        IF
                    old.ak_fertigung
            AND NOT new.ak_fertigung
        THEN
            IF EXISTS
            (
                SELECT true
                FROM opl
                WHERE
                        op_n = new.ak_nr
                    AND op_standard
            )
            THEN
                RAISE EXCEPTION '%', lang_text( 16511 );
            END IF;
        END IF;

        IF new.ak_nr <> old.ak_nr THEN
            PERFORM tartikel.bestand_abgleich( old.ak_nr );

            --bestandsabgleich und bedarfsaktualisierung
            PERFORM tartikel.bestand_abgleich( new.ak_nr );
        END IF;
    END IF;

    IF
        NOT EXISTS
        (
             SELECT true
             FROM artmgc
             WHERE
                     m_ak_nr  = new.ak_nr
                 AND m_mgcode = new.ak_standard_mgc
        )
    THEN
        INSERT INTO artmgc
          (   m_ak_nr,            m_mgcode, m_uf )
        VALUES
          ( new.ak_nr, new.ak_standard_mgc,    1 );

        -- bei Meter automatisch mm wenn Rohmaterial!
        IF
                new.ak_standard_mgc = 4
            AND ic_for_ac( new.ak_ac ) = 3
        THEN

           -- Millimeter
           INSERT INTO artmgc
              ( m_ak_nr, m_mgcode, m_uf )
           VALUES
              ( new.ak_nr, 3, 1000 );

           IF ac_me_code_stange3m FROM artcod WHERE ac_n = new.ak_ac THEN
              INSERT INTO artmgc
                (   m_ak_nr, m_mgcode,               m_uf )
              VALUES
                ( new.ak_nr,       50, 0.3333333333333333 );

           -- Todo: Idee: (DS) Bei Stangen automatisch Stange als ME.
           --   Stange 3m (Stange 6m ist äußerst selten und sollte wenn manuell hinzugefügt werden)
           -- INSERT INTO artmgc (m_ak_nr, m_mgcode, m_uf) VALUES (new.ak_nr, 51, 0.1666666666666667);
           END IF;
        END IF;
    END IF;

    -- Sprachen und Sprachfähigkeit, Artikeleigenschaften
    IF TG_OP = 'INSERT' THEN

        --INSERT
        IF COALESCE( new.ak_hest, 0 ) <> 0 THEN
            PERFORM TArtikel.ain_hest_herkunft__set( new.ak_nr );
        END IF;

        -- Wir haben ein Gewicht und legen sogleich die Mengeneinheit Gewicht an,
        --   wenn GME kg, wurde das eben schon angelegt.
        IF
                new.ak_gewicht IS NOT null
            AND new.ak_standard_mgc <> 2
        THEN
            INSERT INTO artmgc
              (   m_ak_nr, m_mgcode,           m_uf )
            VALUES
              ( new.ak_nr,        2, new.ak_gewicht );
        END IF;
        --
        INSERT INTO artblang
          ( akbl_ak_nr, akbl_spr_key, akbl_txt )
        VALUES
          ( new.ak_nr, prodat_languages.curr_lang(), new.ak_bez );

        INSERT INTO artmlang
          ( akml_ak_nr, akml_spr_key, akml_txt )
        VALUES
          ( new.ak_nr, prodat_languages.curr_lang(), new.ak_mat );

        INSERT INTO artdlang
          ( akdl_ak_nr, akdl_spr_key, akdl_txt )
        VALUES
          ( new.ak_nr, prodat_languages.curr_lang(), new.ak_dim );
        --
    ELSE
        -- UPDATE

        -- Bei Änderung von Gewicht, UF von kg automatisch nachziehen, außer GME = kg (UF von 1kg zu ME bleibt)
        IF
                new.ak_gewicht IS DISTINCT FROM old.ak_gewicht
            AND new.ak_standard_mgc <> 2
        THEN
            IF new.ak_gewicht IS NOT null THEN
                UPDATE artmgc
                SET m_uf = new.ak_gewicht
                WHERE
                        m_ak_nr = new.ak_nr
                    AND m_mgcode = 2
                    AND m_uf IS DISTINCT FROM new.ak_gewicht
                ;
            END IF;

            IF
                    NOT EXISTS
                    (
                        SELECT true
                        FROM artmgc
                        WHERE
                              m_ak_nr = new.ak_nr
                          AND m_mgcode = 2
                    )
                AND ( new.ak_gewicht IS NOT null ) THEN
                INSERT INTO artmgc
                  (   m_ak_nr, m_mgcode,           m_uf )
                VALUES
                  ( new.ak_nr,        2, new.ak_gewicht );
            END IF;
        END IF;
        --
        IF old.ak_bez IS DISTINCT FROM new.ak_bez THEN
            DELETE FROM artblang
            WHERE
                    akbl_ak_nr   = new.ak_nr
                AND akbl_spr_key = prodat_languages.curr_lang();

            INSERT INTO artblang
              ( akbl_ak_nr, akbl_spr_key, akbl_txt )
            VALUES
              ( new.ak_nr, prodat_languages.curr_lang(), new.ak_bez );
        END IF;
        --
        IF old.ak_mat IS DISTINCT FROM new.ak_mat THEN
            DELETE FROM artmlang
            WHERE
                    akml_ak_nr   = new.ak_nr
                AND akml_spr_key = prodat_languages.curr_lang();

            INSERT INTO artmlang
              ( akml_ak_nr,                 akml_spr_key,   akml_txt )
            VALUES
              (  new.ak_nr, prodat_languages.curr_lang(), new.ak_mat );
        END IF;
        --
        IF old.ak_dim IS DISTINCT FROM new.ak_dim THEN
            DELETE FROM artdlang
            WHERE
                      akdl_ak_nr = new.ak_nr
                AND akdl_spr_key = prodat_languages.curr_lang();

            INSERT INTO artdlang
              ( akdl_ak_nr,                 akdl_spr_key,   akdl_txt )
            VALUES
              (  new.ak_nr, prodat_languages.curr_lang(), new.ak_dim );
        END IF;
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__a_iu
    AFTER INSERT OR UPDATE
    ON art
    FOR EACH ROW
    EXECUTE PROCEDURE art__a_iu();
 --

 --
 CREATE OR REPLACE FUNCTION art__as__a_iu() RETURNS TRIGGER AS $$
  BEGIN
    --IF NOT in_artbestandabgleich() THEN
     PERFORM do_artikel_bedarf();
    --END IF;
    RETURN NULL;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__as__a_iu
    AFTER INSERT OR UPDATE
    ON art
    FOR EACH STATEMENT
    EXECUTE PROCEDURE art__as__a_iu();
 --

 -- Prüfplan in ASK erstellen
 CREATE OR REPLACE FUNCTION art__a_u_autopruefplan() RETURNS TRIGGER AS $$
  BEGIN
    IF new.ak_norm = 'DIN EN 9100' THEN
        PERFORM opl_autopruefplan(op_ix , 'DIN EN 9100', FALSE) -- FALSE für Prüfplan in op8 obsolet
        FROM opl WHERE op_n = new.ak_nr;
    END IF;

    IF old.ak_norm = 'DIN EN 9100' THEN
        PERFORM opl_autopruefplan(op_ix , 'DIN EN 9100', TRUE) -- TRUE für Prüfplan in op8 obsolet
        FROM opl WHERE op_n = new.ak_nr
          AND EXISTS(SELECT TRUE FROM op8 WHERE o8_ix = op_ix AND o8_op_stat = 'E');
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__a_u_autopruefplan
    AFTER UPDATE OF ak_norm
    ON art
    FOR EACH ROW
    WHEN (new.ak_norm IS DISTINCT FROM old.ak_norm)
    EXECUTE PROCEDURE art__a_u_autopruefplan();
 --

 --
 CREATE OR REPLACE FUNCTION art__a_iu_kunde() RETURNS TRIGGER AS $$
      BEGIN
        INSERT INTO artzuo (az_pronr, az_prokrz)
        SELECT new.ak_nr, new.ak_kunde
        WHERE NOT EXISTS(SELECT TRUE FROM artzuo WHERE az_pronr = new.ak_nr AND az_prokrz = new.ak_kunde);

        RETURN new;
      END $$ LANGUAGE plpgsql;

      CREATE TRIGGER art__a_iu_kunde
        AFTER INSERT OR UPDATE OF ak_kunde
        ON art
        FOR EACH ROW
        WHEN (new.ak_kunde IS NOT NULL AND new.ak_kunde IS DISTINCT FROM '#')
        EXECUTE PROCEDURE art__a_iu_kunde();
 --
 CREATE OR REPLACE FUNCTION art__a_iu__txtfields() RETURNS TRIGGER AS $$
     BEGIN
        --- # 7715 Zusatztext Artikel Mehrsprachig
        IF tg_op='INSERT' THEN
          IF NOT EXISTS (SELECT true FROM artdokutxt WHERE adtx_ak_nr = new.ak_nr AND adtx_spr_key = prodat_languages.curr_lang()) THEN
            INSERT INTO artdokutxt (adtx_ak_nr, adtx_spr_key, adtx_auftxt, adtx_bestxt, adtx_lieftxt, adtx_faktxt)
             VALUES (new.ak_nr, prodat_languages.curr_lang(), new.ak_auftxt, new.ak_bestxt, new.ak_lieftxt, new.ak_faktxt);
          END IF;
        ELSE
            UPDATE artdokutxt SET
              adtx_auftxt = new.ak_auftxt, adtx_bestxt = new.ak_bestxt, adtx_lieftxt = new.ak_lieftxt, adtx_faktxt = new.ak_faktxt,
              adtx_auftxt_rtf = new.ak_auftxt_rtf, adtx_bestxt_rtf = new.ak_bestxt_rtf, adtx_lieftxt_rtf = new.ak_lieftxt_rtf, adtx_faktxt_rtf = new.ak_faktxt_rtf
             WHERE adtx_ak_nr = new.ak_nr AND adtx_spr_key = prodat_languages.curr_lang();
        END IF;
        --
        RETURN new;
     END $$ LANGUAGE plpgsql;

     CREATE TRIGGER art__a_iu__txtfields
      AFTER INSERT OR UPDATE
      OF ak_auftxt, ak_auftxt_rtf, ak_bestxt, ak_bestxt_rtf, ak_lieftxt, ak_lieftxt_rtf, ak_faktxt, ak_faktxt_rtf
      ON art
      FOR EACH ROW
      EXECUTE PROCEDURE art__a_iu__txtfields();

 --ak_min und ak_melde werden nie in massen geupdated, daher der einfachheit halber so.
 --zusätzlich ist ak_melde eh blödsinn derzeit, das wird nie direkt benutzt für bedarsberechnung
 CREATE OR REPLACE FUNCTION art__a_u_minmelde() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM tartikel.prepare_artikel_bedarf(new.ak_nr, false);
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__a_u_minmelde
    AFTER UPDATE OF ak_min, ak_melde
    ON art
    FOR EACH ROW
    EXECUTE PROCEDURE art__a_u_minmelde();
 --

 -- Auslaufdatum setzen, wenn Artikel aufgebraucht (Otpion ak_auslauf_aufbrauch), bei Aufbrauchen auf ungültig
 CREATE OR REPLACE FUNCTION art__b_u_ak_auslauf_aufbrauch() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (new.ak_tot <= 0 AND new.ak_auslauf_aufbrauch AND new.ak_auslauf IS NULL)
    new.ak_auslauf:=current_date;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__b_u_ak_auslauf_aufbrauch
    BEFORE UPDATE OF ak_tot, ak_auslauf_aufbrauch
    ON art
    FOR EACH ROW
    WHEN (new.ak_tot <= 0 AND new.ak_auslauf_aufbrauch AND new.ak_auslauf IS NULL)
    EXECUTE PROCEDURE art__b_u_ak_auslauf_aufbrauch();
 --

 -- #8887 berücksichtigung Beschaffungsfrist bei Bestellanforderungen
 CREATE OR REPLACE FUNCTION art__a_u_ak_bfr() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM tartikel.prepare_artikel_bedarf(new.ak_nr, false);
    RETURN new;
  END $$ LANGUAGE plpgsql;

 CREATE TRIGGER art__a_u_ak_bfr
    AFTER UPDATE OF ak_bfr
    ON art
    FOR EACH ROW
    EXECUTE PROCEDURE art__a_u_ak_bfr();
--

-- #16611 Gefahrguteigenschaften werden gelöscht, wenn ak_gefahrgut = false
CREATE OR REPLACE FUNCTION art__a_u_ak_gefahrgut() RETURNS TRIGGER AS $$
  BEGIN
      DELETE FROM recnokeyword
      WHERE
              r_tablename = 'art'
          AND r_dbrid = new.dbrid
          AND r_reg_pname LIKE 'ak_gefahrgut%';

      RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__a_u_ak_gefahrgut
    AFTER UPDATE OF ak_gefahrgut
    ON art
    FOR EACH ROW
    WHEN ( new.ak_gefahrgut = false )
    EXECUTE PROCEDURE art__a_u_ak_gefahrgut();
--

--
CREATE TABLE artlog (
  akl_id                SERIAL PRIMARY KEY,
  akl_time              TIMESTAMP(0) WITHOUT TIME ZONE DEFAULT currenttime(), -- Zeitpunkt
  akl_category          VARCHAR(40),            -- Kategorie des Log-Eintrags (insb. für CAD-Steuerung. #7181)
  akl_aknr              VARCHAR(40) NOT NULL,   --Artikelnummer
  akl_ak_bez_old        VARCHAR(100),           -- Artikelbezeichnung
  akl_ak_bez_new        VARCHAR(100),
  akl_ak_znr_old        VARCHAR(50),            -- Zeichnungsnummer
  akl_ak_znr_new        VARCHAR(50),
  akl_ak_idx_old        VARCHAR(30),            -- Zeichnungsindex
  akl_ak_idx_new        VARCHAR(30),
  akl_ak_hest_old       NUMERIC(12,4),          -- Selbstkosten
  akl_ak_hest_new       NUMERIC(12,4),
  akl_ak_vkpbas_old     NUMERIC(12,4),          -- Verkaufspreisbasis
  akl_ak_vkpbas_new     NUMERIC(12,4),
  akl_cad_event_old     VARCHAR(100),           -- CAD-Ereignisse bzgl. Kategorie (CAD-Status, Pxx-Status, #7181)
  akl_cad_event_new     VARCHAR(100),
  akl_txt               TEXT,                   -- langtext für Einträge wie zum Beispiel vorhergehende ak_nr bei Kopierten Artikeln #7441
  akl_is_in_auditlog    BOOLEAN DEFAULT true    -- Dieser Datensatz ist aufgrund des Parallelbetriebs der Logs ebenso im Audit-Log
 );

 --
 CREATE OR REPLACE FUNCTION art__a_u_log() RETURNS TRIGGER AS $$
  DECLARE akbzo VARCHAR;
          akbzn VARCHAR;
          akznr_old VARCHAR;
          akznr_new VARCHAR;
          akidx_old VARCHAR;
          akidx_new VARCHAR;
          akho  NUMERIC;
          akhn  NUMERIC;
          akvo  NUMERIC;
          akvn  NUMERIC;
          doit  BOOL;
  BEGIN
    doit:=FALSE;
    --Artikelbezeichnung
    IF COALESCE(new.ak_bez,'') <> COALESCE(old.ak_bez, '') THEN
        doit:=TRUE;
        akbzo:=old.ak_bez;
        akbzn:=new.ak_bez;
    END IF;
    -- Zeichnungs-Nr.
    IF COALESCE(new.ak_znr,'') <> COALESCE(old.ak_znr, '') THEN
        doit:=TRUE;
        akznr_old:=old.ak_znr;
        akznr_new:=new.ak_znr;
    END IF;
    -- Zeichnungsindex
    IF COALESCE(new.ak_idx,'') <> COALESCE(old.ak_idx, '') THEN
        doit:=TRUE;
        akidx_old:=old.ak_idx;
        akidx_new:=new.ak_idx;
    END IF;
    --Selbstkosten
    IF COALESCE(new.ak_hest,0) <> COALESCE(old.ak_hest, 0) THEN
        doit:=TRUE;
        akho:=old.ak_hest;
        akhn:=new.ak_hest;
    END IF;
    --Verkaufspreisbasis
    IF COALESCE(new.ak_vkpbas,0) <> COALESCE(old.ak_vkpbas, 0) THEN
        doit:=TRUE;
        akvo:=old.ak_vkpbas;
        akvn:=new.ak_vkpbas;
    END IF;
    --falls aenderung, dann eintragen in log-tabelle
    IF doit THEN
        INSERT INTO artlog (akl_aknr, akl_ak_bez_old, akl_ak_bez_new, akl_ak_znr_old, akl_ak_znr_new, akl_ak_idx_old, akl_ak_idx_new, akl_ak_hest_old, akl_ak_hest_new, akl_ak_vkpbas_old, akl_ak_vkpbas_new)
        VALUES (new.ak_nr, akbzo, akbzn, akznr_old, akznr_new, akidx_old, akidx_new, akho, akhn, akvo, akvn);
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER art__a_u_log
    AFTER UPDATE
    ON art
    FOR EACH ROW
    EXECUTE PROCEDURE art__a_u_log();
--

-- Fremdsprachdateien für Artikelstamm
 -- Bezeichnungen
 CREATE TABLE artblang (
   akbl_id      SERIAL NOT NULL PRIMARY KEY,
    akbl_ak_nr       VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,
    akbl_spr_key     VARCHAR(5) NOT NULL CONSTRAINT xtt4089 REFERENCES sprach,
    akbl_txt     VARCHAR(100)
   );

 -- Material
 CREATE TABLE artmlang (
   akml_id      SERIAL NOT NULL PRIMARY KEY,
    akml_ak_nr       VARCHAR(40) NOT NULL CONSTRAINT artmatlang_art REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,
    akml_spr_key     VARCHAR(5) NOT NULL CONSTRAINT xtt4089 REFERENCES sprach,
    akml_txt     VARCHAR(50)
   );

 -- Dimension
 CREATE TABLE artdlang (
   akdl_id      SERIAL NOT NULL PRIMARY KEY,
    akdl_ak_nr       VARCHAR(40) NOT NULL CONSTRAINT artdimlang_art REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,
    akdl_spr_key     VARCHAR(5) NOT NULL CONSTRAINT xtt4089 REFERENCES sprach,
    akdl_txt     VARCHAR(50)
   );

 -- Zusatztexte
 CREATE TABLE arttxt (
  aktxt_ak_nr      VARCHAR(40) NOT NULL PRIMARY KEY REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,
   aktxt_kind       VARCHAR(1),--A=Auftrag B=Bestellung F=Faktura L=Lieferschein
   aktxt_spr_key    VARCHAR(5) NOT NULL REFERENCES sprach,
   aktxt_txt        TEXT
  );
 --

--

-- Mengeneinheiten
CREATE TABLE artmgc (
  m_id          SERIAL NOT NULL PRIMARY KEY,
  m_ak_nr       VARCHAR(40) NOT NULL REFERENCES art ON DELETE CASCADE ON UPDATE CASCADE,
  m_mgcode      INTEGER NOT NULL REFERENCES mgcode ON DELETE RESTRICT,
  m_uf          NUMERIC NOT NULL CHECK(m_uf>0) DEFAULT 1
  );
 --

 CREATE UNIQUE INDEX xtt4098 ON artmgc(m_ak_nr, m_mgcode);

 --
 CREATE OR REPLACE FUNCTION artmgc__b_iu() RETURNS TRIGGER AS $$
 BEGIN
  IF (new.m_id = tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(new.m_ak_nr)) THEN
     IF new.m_uf<>1 THEN
        RAISE EXCEPTION 'xtt21454 %', lang_text(29201) /*standardME UF <> 1!!'*/;
     END IF;
  END IF;
  RETURN new;
 END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artmgc__b_iu
  BEFORE INSERT OR UPDATE
  ON artmgc
  FOR EACH ROW
  EXECUTE PROCEDURE artmgc__b_iu();
 --

 --Standardmengeneinheit darf nicht gelöscht werden
 CREATE OR REPLACE FUNCTION artmgc__b_d() RETURNS TRIGGER AS $$
  BEGIN
    IF (old.m_id = tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(old.m_ak_nr)) AND (NOT (current_user = 'syncro')) THEN
        --Wenn es den Artikel noch gibt unterbinden wir das
        IF EXISTS(SELECT true FROM art WHERE old.m_ak_nr = ak_nr) THEN
            RAISE EXCEPTION 'xtt12019';--StandardME darf nicht gelöscht werden
        END IF;
    END IF;

    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artmgc__b_d
    BEFORE DELETE
    ON artmgc
    FOR EACH ROW
    EXECUTE PROCEDURE artmgc__b_d();
 --

 -- Mengeneinheit der ID darf nicht umgesetzt werden. Sonst werden still zb alle jemals verwendeten Stück zu Meter.
 CREATE OR REPLACE FUNCTION artmgc__b_u() RETURNS TRIGGER AS $$
  BEGIN
    IF current_user <> 'syncro' AND
        (   EXISTS(SELECT TRUE FROM auftg WHERE ag_mcv=new.m_id)
            OR EXISTS(SELECT TRUE FROM belegpos WHERE belp_mce=new.m_id)
            OR EXISTS(SELECT TRUE FROM belzeil_auftg_lif WHERE belzeil_auftg_lif.bz_mce=new.m_id)
            OR EXISTS(SELECT TRUE FROM belzeil_grund WHERE belzeil_grund.bz_mce=new.m_id)
            OR EXISTS(SELECT TRUE FROM bestvorschlagpos WHERE bvp_mcv=new.m_id)
            OR EXISTS(SELECT TRUE FROM epreis WHERE e_mcv=new.m_id)
            OR EXISTS(SELECT TRUE FROM invlag WHERE il_mce=new.m_id)
            OR EXISTS(SELECT TRUE FROM lag WHERE lg_mce=new.m_id)
            OR EXISTS(SELECT TRUE FROM ldsdok WHERE ld_mce=new.m_id)
            OR EXISTS(SELECT TRUE FROM lifsch WHERE l_abg_mec=new.m_id)
            OR EXISTS(SELECT TRUE FROM op6 WHERE o6_mce=new.m_id)
            OR EXISTS(SELECT TRUE FROM rahmenlieferant WHERE rhl_mgc=new.m_id)
            OR EXISTS(SELECT TRUE FROM stv WHERE st_mgc=new.m_id)
            OR EXISTS(SELECT TRUE FROM stvtrs WHERE stmgc=new.m_id)
            OR EXISTS(SELECT TRUE FROM vertrag_pos WHERE vtp_mce=new.m_id)
            OR EXISTS(SELECT TRUE FROM wendat WHERE w_zug_mec=new.m_id)
        )
    THEN
        RAISE EXCEPTION '%', lang_text(16261);
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artmgc__b_u
    BEFORE UPDATE OF m_mgcode
    ON artmgc
    FOR EACH ROW
    EXECUTE PROCEDURE artmgc__b_u();
 --
 CREATE OR REPLACE FUNCTION artmgc__a_iud() RETURNS TRIGGER AS $$
   BEGIN
      IF (tg_op='INSERT') THEN
        IF (new.m_mgcode=2) THEN
          UPDATE art SET ak_gewicht=new.m_uf WHERE ak_nr=new.m_ak_nr AND ak_gewicht IS DISTINCT FROM new.m_uf;
        END IF;
        RETURN new;
      END IF;
      --
      IF (tg_op='UPDATE') THEN
        IF (new.m_mgcode=2 AND new.m_uf<>old.m_uf) THEN
          UPDATE art SET ak_gewicht=new.m_uf WHERE ak_nr=new.m_ak_nr AND ak_gewicht IS DISTINCT FROM new.m_uf;
        END IF;
        RETURN new;
      END IF;
      --

      IF tg_op='DELETE' THEN
        IF (old.m_mgcode=2) THEN
          UPDATE art SET ak_gewicht=NULL WHERE ak_nr=old.m_ak_nr;
        END IF;
        RETURN old;
      END IF;
      --
   END $$ LANGUAGE plpgsql;


    CREATE TRIGGER artmgc__a_iud
      AFTER INSERT OR UPDATE OR DELETE
      ON artmgc
      FOR EACH ROW
      EXECUTE PROCEDURE artmgc__a_iud();



-- Verkaufspreise
CREATE TABLE artvkp (
  vkp_id        SERIAL NOT NULL PRIMARY KEY,
  vkp_aknr      VARCHAR(40) NOT NULL REFERENCES art ON DELETE CASCADE ON UPDATE CASCADE,
  vkp_kukl      SMALLINT CONSTRAINT xtt4088 CHECK(vkp_kukl>0),
  vkp_vkp       NUMERIC(12,4) NOT NULL DEFAULT 0,
  vkp_vkpold        NUMERIC(12,4),
  vkp_waco      VARCHAR(3) NOT NULL DEFAULT TSystem.Settings__Get('BASIS_W') REFERENCES bewa,
  vkp_txt       TEXT,
  vkp_vondat        DATE DEFAULT current_date,
  vkp_bisdat        DATE
 );
 --

 CREATE UNIQUE INDEX xtt499 ON artvkp (vkp_aknr,vkp_kukl, vkp_bisdat);

--
CREATE TABLE artrab (
  rab_id         SERIAL NOT NULL PRIMARY KEY,
  rab_aknr      VARCHAR(40) NOT NULL REFERENCES art ON DELETE CASCADE ON UPDATE CASCADE,
  rab_menge     NUMERIC CONSTRAINT xtt4095 CHECK(rab_menge>=0),
  rab_kukl      INTEGER,
  rab_prozent       NUMERIC NOT NULL
 );
 --

 CREATE UNIQUE INDEX  xtt4070 ON artrab (rab_aknr,rab_menge);

-- Staffelpreise für Artikel, abhängig von Kundenklasse
CREATE TABLE artstp (
  astp_id           SERIAL NOT NULL PRIMARY KEY,
  astp_aknr         VARCHAR(40) NOT NULL REFERENCES art ON DELETE CASCADE ON UPDATE CASCADE,
  astp_menge        NUMERIC NOT NULL,
  astp_kukl         SMALLINT CONSTRAINT xtt4088 CHECK(astp_kukl>0),
  astp_vkp          NUMERIC CONSTRAINT xtt4096 CHECK(astp_vkp>0),
  astp_waco         VARCHAR(3) NOT NULL DEFAULT TSystem.Settings__Get('BASIS_W') REFERENCES bewa,
  astp_fixpreis     BOOL NOT NULL DEFAULT FALSE,
  astp_datgv        DATE,
  astp_datgb        DATE
 );
 --

 CREATE UNIQUE INDEX artstp__ustk ON artstp(astp_aknr, astp_kukl, astp_menge);

--ACHTUNG CNC-Spezifischer Trigger "artstp__a_99_iud__updateauftg__cnc" für aktualisieren Staffeln in Angeboten bei Änderungen im Artikel => TotalCustomer

--
CREATE TABLE artoption_arts (
  aoa_id       SERIAL PRIMARY KEY,
  aoa_pos      INTEGER,
  aoa_g_ak_nr  VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE, -- grundartikel
  aoa_ak_nr    VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE, -- optionsartikel
  CONSTRAINT aoa_g_ak_nr__aoa_ak_nr__chk CHECK ( aoa_g_ak_nr != aoa_ak_nr ) -- Artikel darf nicht seine eigene Alternative sein
 );

--Zusatzinformationen zum Artikel, die Art soll nich weiter aufgeblaeht werden
CREATE TABLE artinfo (
  ain_ak_nr         varchar(40) PRIMARY KEY REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,
  ain_canSkonto     boolean NOT NULL DEFAULT TRUE,  -- Artikel ist skontofaehig?
  ain_steucode_ek   integer REFERENCES steutxt,     -- Artikel hat Standardsteuersatz?
  ain_lagzu_txt     text,                           -- Einlagerungshinweise. Wird direkt vor Lagerzugang angezeigt.
  ain_lagzu_txt_rtf text,
  ain_fixbfr        boolean NOT NULL DEFAULT FALSE, -- Beschaffungsfrist wird nicht von Lieferantendaten aktualisiert
  ain_ak3code       varchar(3),                     -- 3-stelliger Artikel-Code //Ticket 5203
  ain_vpe           numeric,           -- VorgabeMenge pro Verpackungseinheit in Standardmengeneinheit // #5975
  ain_vpe_tol       numeric,           -- Toleranzmenge (pro Lagerabgang iVm Verpackungseinheit) in Standardmengeneinheit // #5975
  ain_hest_datum    date,                    -- Datum (Selbstkosten)
  ain_hest_herkunft varchar(100),            -- Ursprung (Selbstkosten)
  ain_hest_user     varchar(50),
  ain_minmenge      numeric,                 -- Mindestmenge Rahmenauftrag #5381#note-25
  ain_minmengeproz  numeric,                 -- Mindestmenge Prozentual Rahmenauftrag #5381#note-25
  ain_op_ix         integer,                  -- Zugewiesene Standard-ASK (opl.op_ix) -- FKEY siehe TableContraints.sql
  ain_lue_req       boolean                  -- Ursprungserklärung erforderlich #7934
 );
 --

 --
 CREATE OR REPLACE FUNCTION artinfo__b_iu_ak3code() RETURNS TRIGGER AS $$
  DECLARE aknr VARCHAR;
  BEGIN
    --Prüfen ob Code 3 Stellig und Numerisch
    IF (new.ain_ak3code !~* E'^\\d{3}$') AND COALESCE(new.ain_ak3code, '')<>'' THEN
        RAISE EXCEPTION '% %', lang_text(26038), lang_text(26039);
    END IF;
    --Prüfen ob Kode einmalig vorkommt
    SELECT ain_ak_nr INTO aknr FROM artinfo WHERE ain_ak3code=new.ain_ak3code ORDER BY ain_ak_nr LIMIT 1;
    IF aknr IS NOT NULL THEN
        RAISE EXCEPTION '% % %', new.ain_ak3code, lang_text(26044),  E'\n\n'||aknr;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artinfo__b_iu_ak3code
    BEFORE INSERT OR UPDATE
    OF ain_ak3code
    ON artinfo
    FOR EACH ROW
    WHEN (new.ain_ak3code IS NOT NULL)
    EXECUTE PROCEDURE artinfo__b_iu_ak3code();

-- ************************** Stücklistenstruktur / Baumaufbau von Artikel- und Arbeitspaketstücklisten ******************************
CREATE TABLE stv (
  st_id                 serial NOT NULL PRIMARY KEY,
  st_zn                 varchar(40) NOT NULL REFERENCES art ON UPDATE CASCADE, -- Parentartikel / Übergeordnetes Element aus Ebene N
  st_n                  varchar(40) NOT NULL REFERENCES art ON UPDATE CASCADE, -- Unterartikel der Ebene N-1
  st_pos                smallint NOT NULL,                      -- Sortierung / Reihenfolge der Artikel einer Ebene
  st_pos_parent         smallint,                               -- Verflachte Stücklisten: Paren merken. #9515
  st_mgc                integer  NOT NULL CONSTRAINT xtt5015 REFERENCES artmgc,-- Mengeneinheit für den Unterartikel
  st_m                  numeric NOT NULL,                       -- Anzahl der Unterartikel, die in Parentartikel enthalten sind
  st_m_uf1              numeric,                                -- Anzahl der Unterartikel, in Grund-ME
  st_m_fix              boolean NOT NULL DEFAULT FALSE,         --Fixmenge, wird in Stücklistenauflösungen nicht berechnet
  st_txt                text,
  st_stat               varchar(3),                             --Status Stücklistenteil: BK=Beistellung durch Kunde, BL=Beistellung an Lieferant
  -- st_inv                varchar(20),
  -- st_notinv             varchar(20),
  st_ekenner_krz        varchar(20),
  st_kstv_st_id         integer REFERENCES stv,                 --Fertgungsstückliste: Bezug zu (Konstruktions-)Stückliste, wo Teil herkommt
  CONSTRAINT xtt5021 CHECK (not(st_n=st_zn)),
  st_posinfo            varchar(20)                             -- #8931  Zusatzinformationen zur Positionsnummer
 );
 --

 --
 CREATE OR REPLACE FUNCTION stv__b_iu() RETURNS trigger AS $$
  BEGIN
   new.st_m_uf1 := tartikel.me__menge__in__menge_uf1(new.st_mgc, new.st_m);

   DELETE FROM stvtrs WHERE resid=new.st_zn AND stv_str_revnr IS NULL;/*Änderung der Baugruppe, es muss alles neu aufgebaut werden! Alte Revisionen bleiben bestehen.*/

   --Konstruktions/Fertigungsstückliste: Änderung von Konstruktionsstückliste an Fertigungsstückliste weitergeben.
   IF tg_op='UPDATE' THEN
        IF new.st_n<>old.st_n AND new.st_kstv_st_id IS NOT NULL THEN
           IF new.st_n<>(SELECT st_n FROM stv WHERE st_id=new.st_kstv_st_id) THEN --stv__a_iud ansonsten kam die Änderung von dort, und da steht schon die neue Artikelnr drin
                new.st_kstv_st_id:=NULL;
           END IF;
        END IF;
   END IF;

   --PERFORM * FROM tartikel.stueckl__do_stueckl_list('tmp', new.st_zn, 1, null, true);--prüfung ob zirkulär, dauert zu lange und funktioniert auch nicht richtig! neu bauen!!! DS 2011-02-02

   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER stv__b_iu
  BEFORE UPDATE OR INSERT
  ON stv
  FOR EACH ROW
  EXECUTE PROCEDURE stv__b_iu();
 --

 --
 CREATE OR REPLACE FUNCTION stv__a_iud() RETURNS trigger AS $$
  DECLARE _aknr VARCHAR;
          r RECORD;
  BEGIN
   --Kopfdatensatz nachträglich anlegen
   IF (TG_OP = 'INSERT') OR (TG_OP = 'UPDATE') THEN
        IF NOT EXISTS(SELECT true FROM stvzutxt WHERE stv_zn = new.st_zn) THEN
            INSERT INTO stvzutxt (stv_id,stv_zn) VALUES ((SELECT nextval('stvzutxt_stv_id_seq')),new.st_zn);
        END IF;
    END IF;
   --Variable zur Weiterbearbeitung in Abhängigkeit von new/old/insert/delete
   IF (TG_OP = 'DELETE') THEN
        _aknr:=old.st_zn;
    ELSE
        _aknr:=new.st_zn;
    END IF;
   --Modified setzen
   UPDATE stvzutxt SET stv_zn = stv_zn WHERE stv_zn=_aknr;
   --Konstruktions/Fertigungsstückliste: Änderung von Konstruktionsstückliste an Fertigungsstückliste weitergeben.
   IF tg_op='UPDATE' THEN
        IF new.st_n<>old.st_n THEN
            UPDATE stv SET st_n=new.st_n, st_mgc=new.st_mgc WHERE st_kstv_st_id=new.st_id;--ACHTUNG stv__b_iu
        END IF;
    END IF;
   --Cache für Prognoseaufträge entfernen
   DELETE FROM stvtrs WHERE resid IN (SELECT DISTINCT resid FROM stvtrs WHERE resid LIKE 'VALID-1%' AND stvtrs.aknr=_aknr); --wir suche alle Stücklistenauflösungen, welche unseren Kopfartikel, welcher gerade geändert wurde, enthalten
   --
   RETURN null;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER stv__a_iud
  AFTER INSERT OR UPDATE OR DELETE
  ON stv
  FOR EACH ROW
  EXECUTE PROCEDURE stv__a_iud();
--

-- Stücklisten-Typen gemäß der Anwendungsfälle
  -- z.B. BGr-STD = Standard-Baugruppe ; BL = Beistellung an Lieferanten; PS = Produktions-Set
CREATE TABLE stv_typ (
  stt_ident   varchar(10) NOT NULL PRIMARY KEY, -- Identifikator
  stt_bez     varchar NOT NULL  -- Bezeichnung
);

--
CREATE TABLE stvzutxt (
  stv_id                SERIAL NOT NULL PRIMARY KEY,
  stv_zn                VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE, -- Gruppenartikelnummer
  stv_txt               TEXT,                                                                    -- Hinweise zur Stückliste
  stv_stat              VARCHAR(10) NOT NULL DEFAULT 'BGr-STD' REFERENCES stv_typ ON UPDATE CASCADE ON DELETE RESTRICT, -- Typ der Stückliste
  stv_isfertigungsstv   BOOLEAN NOT NULL DEFAULT FALSE
);

CREATE UNIQUE INDEX stvzutxt_stv_zn ON stvzutxt (stv_zn);

CREATE OR REPLACE FUNCTION stvzutxt__b_iu() RETURNS TRIGGER AS $$
  BEGIN
   IF new.stv_isfertigungsstv THEN
      new.stv_stat:='PS';
   END IF;
   IF tg_op='UPDATE' THEN
      IF new.stv_stat='PS' AND NOT new.stv_isfertigungsstv AND old.stv_isfertigungsstv THEN
         new.stv_stat:=NULL;
      END IF;
   END IF;
   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER stvzutxt__b_iu
   BEFORE INSERT OR UPDATE
   ON stvzutxt
   FOR EACH ROW
   EXECUTE PROCEDURE stvzutxt__b_iu();

--
CREATE TABLE stv_vari (
  stvv_id       SERIAL PRIMARY KEY,
  stvv_st_id    INTEGER NOT NULL REFERENCES stv ON DELETE CASCADE,
  stvv_zn       VARCHAR(750), --References art: Semmikolonseparierte Liste von Kopfartikeln
  stvv_v        VARCHAR(75),
  stvv_txt      TEXT
 );

 CREATE FUNCTION stv_vari__b_iu() RETURNS TRIGGER AS $$
  BEGIN
   IF EXISTS (SELECT true FROM stv_vari WHERE stvv_st_id=new.stvv_st_id AND stvv_id<>new.stvv_id AND stvv_v=new.stvv_v) THEN
      RETURN null;--dieser Eintrag existiert bereits > verwerfen
   END IF;
   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER stv_vari__b_iu
   BEFORE INSERT OR UPDATE
   ON stv_vari
   FOR EACH ROW
   EXECUTE PROCEDURE stv_vari__b_iu();
--

-- Stücklistennachweis als PDF Export. Stückliste wird als geändert markiert. Durch den Export des Änderungsnachweis (PDF) wird die Markierung wieder entfernt.
CREATE TABLE stv_changes (
  stc_zn        VARCHAR(40) UNIQUE NOT NULL REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,
  stc_time      TIMESTAMP(0) DEFAULT currenttime()
 );

 --
 CREATE OR REPLACE FUNCTION stv__a_iud_savechanges() RETURNS TRIGGER AS $$
  BEGIN
  IF (tg_op='INSERT') OR (tg_op='UPDATE') THEN
    INSERT INTO stv_changes (stc_zn) SELECT new.st_zn WHERE NOT EXISTS (SELECT 1 FROM stv_changes WHERE stc_zn = new.st_zn);
  END IF;

  IF (tg_op='DELETE') OR (tg_op='UPDATE') THEN
    INSERT INTO stv_changes (stc_zn) SELECT old.st_zn WHERE NOT EXISTS (SELECT 1 FROM stv_changes WHERE stc_zn = old.st_zn);
  END IF;

  RETURN new;
  END $$ LANGUAGE plpgsql;

 CREATE TRIGGER stv__a_iud_savechanges
  AFTER INSERT
  OR UPDATE OF st_n, st_m, st_mgc
  OR DELETE
  ON stv
  FOR EACH ROW
  EXECUTE PROCEDURE stv__a_iud_savechanges();

--
CREATE TABLE ersatzteilkenner (
  ekenner_krz        VARCHAR(10) PRIMARY KEY,
  ekenner_bez        VARCHAR(40) NOT NULL
 );
--

--Zusatzdaten zur Stücklisten-Revision
CREATE TABLE stvrevision (
  str_id       SERIAL NOT NULL PRIMARY KEY,
  str_revnr    VARCHAR(40) NOT NULL,  --Revisionsnummer
  str_resid    VARCHAR(50) NOT NULL,  --Artikelnummer, Hauptartikel aus stvtrs
  str_txt      TEXT,                  --Revisionstexte
  str_txt_rtf  TEXT,
  str_default  BOOLEAN DEFAULT FALSE  --Revision die bei Default genommen wird
 );
 --

 CREATE UNIQUE INDEX str_resid_revnr ON stvrevision(str_revnr, str_resid);

 CREATE OR REPLACE FUNCTION stvrevision__a_u() RETURNS TRIGGER AS $$
  BEGIN
   IF new.str_default THEN
     UPDATE stvrevision SET str_default=false WHERE str_resid=new.str_resid AND str_id<>new.str_id AND str_default;
   END IF;
   RETURN new;
  END $$ LANGUAGE plpgsql;

 CREATE TRIGGER stvrevision__a_u
  AFTER UPDATE
  ON stvrevision
  FOR EACH ROW
  EXECUTE PROCEDURE stvrevision__a_u();
--

-- Fehlerprotokoll: Baugruppenkalkulation
CREATE TABLE stvtrs_res_log (
  strl_id               SERIAL PRIMARY KEY,
  strl_resid            VARCHAR(50),    -- Auflösungs-ID, vgl. stvtrs
  strl_res_object       VARCHAR(100),   -- Objekt in der Auflösung, bei dem es Probleme gibt
  strl_res_object_type  VARCHAR(20),    -- Typ des Problemobjects (z.b. Artikel bei Artikelpreis-Problemen)
  strl_issue            VARCHAR(100)
 );
 --


 -- Prüfung und Erstellung der Protokolleinträge
 CREATE OR REPLACE FUNCTION create_stvtrs_res_log(IN in_resid VARCHAR, IN in_res_object VARCHAR, IN in_res_object_type VARCHAR, VARIADIC params VARCHAR[]) RETURNS VOID AS $$
  DECLARE issue_text TEXT;
  BEGIN
    -- Artikelpreis-Probleme
    IF in_res_object_type = 'art_ekpreis' THEN
        -- Parameter 1 = Preis; Parameter 2 = Preisquelle (TWawi.Search_EKPreis.source_table); Parameter 3 = evtl. Fertigungsartikel
        IF params[1] IS NULL OR in_resid LIKE '%_bgkost'THEN -- Preis unbekannt, also raus. Fall gibt es in der stvtrs-Rekursion; Aufruf mit _bgkost aus AVOR ausschließen
            RETURN;
        END IF;

        IF params[1]::NUMERIC = 0 THEN
            issue_text:= 'P0'; -- Preis ist 0
        ELSIF TSystem.Settings__GetBool('BGCALC_EPREIS_STAFF') AND params[2] IN ('art', 'ask') THEN -- nur bei Berücksichtigung der Staffelpreise in Baugruppenkalkulation protokollieren, ob Preise aus Stammdaten kommen.
            issue_text:=params[2]; -- Preis aus Stammdaten (Artikelstamm, AVOR-Stammkarte)
        END IF;

        INSERT INTO stvtrs_res_log (strl_resid, strl_res_object, strl_res_object_type, strl_issue)
        SELECT in_resid, in_res_object || COALESCE(' | ' || params[3], '') /* evtl. Fertigungsartikel bei AW-Paket ausweisen: 'AW.EXTERN | ART' */, 'art_ekpreis', issue_text
        WHERE in_resid <> ''
          AND issue_text IS NOT NULL
          AND NOT EXISTS(SELECT true FROM stvtrs_res_log WHERE strl_resid = in_resid
                                                           AND strl_res_object = in_res_object || COALESCE(' | ' || params[3], '') -- Artikel pro Res nicht doppelt. Preis ist pro Auflösung immer gleich.
                                                           AND strl_res_object_type = 'art_ekpreis');
    END IF;

    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

-- STV - TreeStructure?
CREATE TABLE stvtrs (
  --pkey          SERIAL PRIMARY KEY,
  resdate       date DEFAULT current_date,
  id            integer,            -- ID des Artikels innerhalb einer bestimmten Auflösung. Beim Auflösen von 'stvtrsid_seq' vergeben. Unique innerhalb einer Auflösungs-ID
  resid         varchar(50),        -- Auflösungs-ID einer bestimmten Stücklistenauflösung. Einfachste Fall: Aknr vom Hauptartikel, Sonst: Bsp. Aknr + Kennzeichen was für eine Auflösung (z.Bsp. Abk-Index oder Prognosekennzeichen)
                                    -- Alle Unterteile / Materialien / Datensätze einer Auflösung haben gleiche Resid
  parent        integer,            -- Jeweiliger vorgeordneter Artikel (stvtrs.id)
  parent_stvtrs_dbrid varchar(100), -- stvtrs.dbrid des Parent
  ebene         integer,            -- Hierachiestufe
  pos           integer,            -- Position Stückliste (kann mehrfach vorkommen) > NEUER TEXT "Pos (Stückliste)"
  aknr          varchar(40),        -- Artikel-Nr.

  kalku         boolean DEFAULT TRUE, -- Kennzeichen ob in der Auflösung kalkuliert werden soll oder nicht für Trigger stvtrs__b_i__calcprices
  romat         boolean DEFAULT FALSE, -- Kennzeichen Rohmaterial
  romat_o6_id   integer,            -- ID von OP6 zum Holen der Zuschnittsangaben
                                    -- ??? abk__auftgi__from_op6__create benötigt keine stvtrs. wann wird das wie gesetzt und genutzt?

  romatp        numeric(16,4),      -- Preis Rohmaterial aus ASK /Kalkulationspreis
  optart        boolean DEFAULT FALSE, -- Kennzeichen Optionsartikel
  stmgc         integer REFERENCES artmgc ON UPDATE CASCADE ON DELETE SET NULL, --Beim Löschen der ME bzw. des Artikel kann die Stkl-Auflösung bzw. Revision geleert werden (nicht stv)
  stm           numeric,            -- Menge des Artikels
  --delete:
  --inv            varchar(20),
  --notinv         varchar(20),
  --Kosten
  selbstko      numeric(16,4),
  vkpbas        numeric(16,4),
  roma          numeric(16,4),      --Rohmaterialkosten
  montage       numeric(16,4),      --Montagekosten
  ruest         numeric(16,4),      --Rüstkosten
  auswaerts     numeric(16,4),      --Auswärtskosten
  --für Baugruppenkosten
  selbstkobg    numeric(16,4),
  vkpbasbg      numeric(16,4),
  romabg        numeric(16,4),      --Rohmaterialkosten
  montagebg     numeric(16,4),      --Montagekosten
  ruestbg       numeric(16,4),      --Rüstkosten
  auswaertsbg   numeric(16,4),      --Auswärtskosten
  --
  selbstko_source_table varchar(50),--Preisherkunft bei Kaufteilen aus Funktion TWawi.Search_EKPreis
  glob_zuschlag_faktor numeric(16,4), -- globale zuschlagskalkulation => faktor. siehe stvtrs__b_i__calcprices
  --
  askix         integer,            -- Stammkartenindex der aufgelösten ASK-Variante

  abkix         integer,            -- ????????

  fertzeit      numeric(16,4),      -- ?????

  stv_dbrid     varchar(100),       --Verbindung zu stv-Datensatz
   -- romat_o6_id => siehe oben
  auftg_ag_id   integer,            --Verbindung zu internem Auftragsdatensatz: bei Auslösen ABK wird Materialliste ausgelöst und jeweils hier die id der Materiallistenposition gespeichert
  stv_str_revnr varchar(40)         --Revisionsnummer, NULL ist momentan Arbeitsstückliste
 );

-- Trigger, welcher für Baugruppenkalkulation die Preise in die STV lädt.
-- BEFORE INSERT (nicht mehr gültig: OR UPDATE, damit nachträglich noch ein ASK-Index umgeschrieben werden kann und die Preise automatisch neu gezogen werden)
CREATE OR REPLACE FUNCTION stvtrs__b_i__calcprices() RETURNS trigger AS $$
  DECLARE
      vkp_faktor            numeric;
      vkpfix                numeric;
      art_is_fertigung      boolean;
      art_is_ic10           boolean;
      r                     record;
      _glob_zuschlag_faktor numeric;
      _parent_ask_index     integer;
      _stv_type             varchar;
      _per_preissuche       boolean := TSystem.Settings__GetBool( 'BGCALC_EPREIS_STAFF' );
  BEGIN
    -- Preise für Baugruppenkalkulation berechnen.


    -- Beistellung durch Kunden (BK) muss mit Wert 0 eingehen, egal ob Stückliste, Fertigungsteil, Vorprodukt oder Material, siehe #11313.
    IF
/*STATUS#20827...*/--TODO hier muss eigentlich "Übergeordnet ist nicht KS und oder AM rein!"
        -- Stücklisten-Pos ist BK
            EXISTS( SELECT true FROM stv WHERE dbrid = new.stv_dbrid   AND TSystem.ENUM_ContainsValue( st_stat, 'BK,AM' ) )
        -- ASK-Materialliste-Pos ist BK
        OR  EXISTS( SELECT true FROM op6 WHERE o6_id = new.romat_o6_id AND TSystem.ENUM_ContainsValue( o6_stat, 'BK,AM' ) )

        -- Mein Parent ist KS, somit habe ich selbst keine Preise mehr. Ich werde "Nur zur Info" aufgelöst
        -- Funktioniert nicht rekursiv. Hier muss der Status korrekt durchgegeben werden in der STV
        -- OR  SELECT st_stat FROM stvtrs JOIN stv ON dbrid = new.stv_dbrid   WHERE id = new.parent
        -- OR  SELECT o6_stat FROM stvtrs JOIN op6 ON o6_id = new.romat_o6_id WHERE id = new.parent
    THEN

        new.selbstko := null;
        new.vkpbas   := null;

        -- weitere Berechnungen für BK nicht mehr nötig.
        RETURN new;

    END IF;


    -- Artikelpreise und -Fertigungstatus ermitteln
    SELECT ak_vkpbas,  ak_hest,      'art',                     ak_vkpbasfix, coalesce( ak_vkpfaktor, ac_vkpfaktor ), ak_fertigung,     ac_i = 10
    INTO   new.vkpbas, new.selbstko, new.selbstko_source_table, vkpfix,       vkp_faktor                            , art_is_fertigung, art_is_ic10
    FROM art
      JOIN artcod ON ak_ac = ac_n
    WHERE ak_nr = new.aknr
    ;


    -- VKP-Faktor auf 1, wenn leer oder 0
    IF coalesce( vkp_faktor, 0 ) = 0 THEN vkp_faktor := 1; END IF;


    _stv_type :=
          stv_stat
        FROM stvzutxt
        WHERE stv_zn = new.aknr
    ;

    -- Beistellung an Lieferanten
    IF _stv_type = 'BL' THEN

        new := tartikel.stvtrs_kalk__beistellung_an_lieferanten( new, _per_preissuche );

    -- Fertigungsartikel (Unterbauteil) mit ASK
    ELSIF new.askix IS NOT NULL THEN

        -- Herkunft der Selbstkosten aus ASK
        new.selbstko_source_table := 'opl';

        -- Kosten pro Stück anhand Kalkulationsmenge
        new.ruest     := tartikel.op2_rkost( new.askix, new.stm );
        new.montage   := tartikel.op2_mkost( new.askix );
        new.auswaerts := tartikel.op2_akost( new.askix, new.stm );
        new.roma      := tartikel.op6_kost(  new.askix, new.stm );

        -- globale Zuschläge (Faktor) der ASK
        _glob_zuschlag_faktor :=
              sum( o7zk_proz ) / 100
            FROM op7zko
            WHERE o7zk_ix = new.askix
              AND o7zk_proz IS NOT NULL
        ;

        _glob_zuschlag_faktor := 1 + coalesce( _glob_zuschlag_faktor, 0 );

        -- Selbstkosten pro Stück = Herstellkosten pro Stück + Globale Zuschläge
        new.selbstko  := ( new.ruest + new.montage + new.auswaerts + new.roma ) * _glob_zuschlag_faktor;

        -- Kosten gesamt anhand Kalkulationsmenge
        new.ruest     := new.ruest     * new.stm;
        new.montage   := new.montage   * new.stm;
        new.auswaerts := new.auswaerts * new.stm;
        new.roma      := new.roma      * new.stm;

    ELSIF (SELECT true FROM stvzutxt WHERE stv_zn = new.aknr AND TSystem.ENUM_GetValue(stv_stat, 'PS') ) THEN

        new.selbstko_source_table := 'SET';
        new.selbstko := null;
        new.vkpbas   := null;

    -- Ausgangspunkt dieses Knotens: was hier hinein fällt, geht nicht in die Kosten ein,
    -- ist durch Festpreis oder z.B. Rohmaterial in ASK usw. evtl. im Überknoten drin, oder ist z.B. "nur zur Konstruktionsansicht".
    ELSIF
        -- Rohmaterial darf keine Preise in Selbstkosten/VKP-Bas haben, da diese sonst mit summiert werden! (Rohmaterialkosten kommen durch Fertigungsteil)
            new.romat

        -- bis 2019-02-15: OR art_is_fertigung allein führt zu massiven bug, daher alt wiederhergestellt #11713:
          -- Produktionsteil, was aktuell eingekauft wird und nur den Haken ak_fertigung gesetzt hat, aber keine ASK kennt.
        OR  (
                art_is_fertigung
            AND NOT art_is_ic10
            -- wenn ic10, gehen wir implizit von SET aus, wenn keine ASK aber Stückliste
            AND EXISTS(SELECT true FROM stv WHERE st_zn = new.aknr)
        )
        -- Gleiches gilt, wenn ich selbst eine Baugruppe (Fertigung) ohne ASK bin. Bei Kaufteilen mit Beistellung durch Lieferanten muss gerechnet werden! #9493
        -- ak_fertigung darf nicht blind berücksichtigt werden, da ein Kaufteil was eigentlich Fertigungsteil war, sonst in Selbstkosten NULL führt (es gibt ja keine Stückliste wie bei Beistellung),
        -- daher AND Stückliste, was auch die Beistellung abbilden sollte, das ist zu prüfen > #11713, #9493
        -- Loll: KHU5160FCD41'A' enthält Fertigungsartikel (Haken gesetzt), die extern eingekauft werden (keine ASK) > daher ist "ak_fertigung" nicht eindeutig
        -- Loll: LO005083'-'00 enthält einen Artikel mit Stückliste, Artikel ist kein Fertigungsartikel (ak_fertigung NICHT gesetzt) aber eine Stückliste [Fall Divers]
        --
        -- Rohmaterial
        -- Montageartikel/Sets (diese ergeben sich aus den Preisen der Unterbauteile, ohne das es eine ASK gibt)
        -- Beachte, das das Flag "isromat" auch von außen genutzt wird, um diesen Knoten anzusprechen, auch bei NICHT Romat (zB "Nur zur Konstruktionsansicht")
        --
        -- Kaufteil mit Beistellung, Stückliste mit Unterstückliste ohne definierten Zustand
        -- Der Preis des Kopfartikels in den Stammdaten ist der Komplettpreis - für Lagerbestände usw.
        -- In der BG-Kalkulation wird bei Kaufteilen auf dieser Ebene aber die Preissuche verwendet, um Einkaufspreis zzgl Unterbauteile zu erreichen
        -- Ist kein Lieferantendatensatz vorhanden, nehmen wir ????? 0 > führ dazu das der Preis entfällt und nur die Unterbauteile gerechnet werden > Selbskosten > dann dürfen keine Unterbauteile gerechnet werden
        -- Wenn Kopf als Set definiert ist, dann Unterbauteile, sonst werden Unterbauteile als Rohmaterial markiert und die Selbstkosten/Einkaufspreis des Artikels als gesamtes betrachtet (kein Set, sondern eingekaufte Stückliste)
        --
        -- Heißt: möchte ich, das Kaufteil mit Beistellung richtig berechnet wird, muß Kopfartikel Fertigungsartikel richtig geflaggt sein.
        --
        -- BGKost = Nur Hauptartikel (selbstko)   [Produktion]              => Fixpreis (unterbauteile werden ignoriert)
        --                                        [Divers]                  => ak_fertigung FALSE ABER stückliste [gegenteil von Set]; entspricht inaktiver Stückliste und wird dann rot dargestellt (als romat markiert)
        --                                        [ProduktionEK]            => ak_fertigung TRUE, keine ASK, keine Stückliste => dann wäre Einkauf mit Beistellung ODER Stückliste => Ich bin Fertigungsartikel UND habe Stückliste ABER keine ASK => gleicher Ablauf wie Produktion EK => FEHLER: ohne Lieferant kein Preis aber auch nicht Lagerwert zzgl Unterbauteile; Widerspruch: Einkauf mit Beistellung => Führt aktuell zu NULL (nicht ask then ak_fertigung and stv setlbstko:=NULL)
        -- BGKost = Summe der Unterbauteile       [Produktion]              => ak_fertigung TRUE, ask vorhanden, unterteile werden addiert
        --                                        [Set]                     => ask nicht vorhanden, Unterstücklisten vorhanden UND ak_fertigung gesetzt [Gegenteil von Divers/Inaktive Unterstückliste]
        --                                                                     Status SET in stv_stat? => entfernt eigene Kosten des Set, diese wären kalkulatorisch im Lager die Summe der Unterteile
        --                                                                     ist Status SET nicht gesetzt =>
        -- BGKost = Kaufpreis + Unterbauteile     [Einkauf mit Beistellung] => Selbstkosten aus Artikelstamm = Lagerpreis komplett => nicht in BGCalc verwendbar
        --                                                                  => Preissuche (Kaufpreis bei Lieferant) zzgl Unterbauteile => ak_fertigung FALSE > widerspricht divers
        --                                                                  => Entspricht ProduktionEK
        --
        -- Grundsätzlicher Widerspruch: ich kaufe ein Fertigungsartikel ein: dargestellt über ak_fertigung=true aber keine ASK vs. Bestellung mit Beistellung = halber Fertigungsartikel aber kein ak_fertigung wegen Widerspruch mit Set
        --
        -- >>> ak_fertigung true und keine Stückliste ist gleicher Fall wie ak_fertigung false und egal => führt zu Preissuche => könnte Selbstkosten oder bei Stammlieferant eben dieser werden !!! Fehlerquelle > Stückliste mit Beistellung, Stammlieferant entfernen > zurückfallen auf [Divers] nötig (ak_fertigung ist false, undefinierter Zustand) > und nehmen selbstkosten Kopfartikel = Lagerwert
        -- >>> gelöschte Preise nur bei: Set [nur Addition der Unterteile], Rohmaterial [in sich kein Wert dieses Unterknotens] [von außen bei Fixpreis eingesteuert für alle Unterknoten]
    THEN

        -- Undefinierten Zustand hinterlegen: kein Rohmaterial aber Fertigungsartikel ohne ASK
        IF NOT new.romat AND art_is_fertigung THEN

            new.romat := true;
            new.selbstko_source_table := 'UNDEFINED DATASTATE';

        END IF;

        -- Zurücksetzen von Selbstkosten und Verkaufspreisbasis, da zu Funktionsbeginn pauschal alles gesetzt wird.
        new.selbstko := null;
        new.vkpbas   := null;


    -- Preise anhand Preissuche ermitteln
      -- ansonsten bleibt der Preis leer und wird weiter unten gesetzt.
    ELSIF _per_preissuche THEN

        new := tartikel.stvtrs_kalk__by__search_ekpreis( new );

    END IF;


    -- Festpreis im Artikelstamm
    IF TSystem.Settings__GetBool( 'bgcalc_vkpfix:' || current_user ) THEN
        -- dann die Selbstkosten rückwärts ausrechnen.
        new.selbstko := coalesce( vkpfix / vkp_faktor, new.selbstko );

    END IF;

    -- Wir haben nur VKP-Basis im Artikelstamm eingetragen.
    -- #19939 Selbstkostenübernahme nicht für Fertigungsartikel, sonst ggf. Doppelberechnung von Baugruppenkosten,
    --        wenn Unterbaugruppen ohne Selbstkosten
    IF new.selbstko = 0 AND new.vkpbas > 0 AND NOT art_is_fertigung THEN

        -- Um das ganze etwas sinniger zu machen, rechnen wir die Selbstkosten zurück.
        new.selbstko := new.vkpbas / vkp_faktor;

    END IF;

    -- Selbstkosten gesamt anhand Kalkulationsmenge
    new.selbstko := new.selbstko * new.stm;


    -- Norm- bzw. Kaufteile (N) der Stückliste mit globalen Zuschlägen der übergordnete ASK versehen.
      -- Ermittlung Materialtyp N analog auftg__b_iu__aknr__stati
      -- Berechnung der globalen Zuschläge analog tabk.get_abk_kosten bzw. tplanterm.buchrm
    IF
        -- Selbstkosten sind vorhanden und nicht null
            new.selbstko > 0
        -- Artikel ist nicht IC 10
        AND NOT art_is_ic10
        -- Artikel ist keine eigene Fertigung
        AND new.askix IS NULL
        -- Artikel ist kein Rohmaterial
        AND NOT new.romat
    THEN

        -- https://ci.prodat-sql.de/sources/tests/suite/37/runner/935
        WITH RECURSIVE
          _tree AS (

            SELECT stvtrs.id,
                   stvtrs.parent,
                   stvtrs.askix
              FROM stvtrs
             WHERE resid = new.resid
               AND id = new.parent -- wir selbst sind ausgangspunkt
               AND stv_str_revnr IS NULL

            UNION

            SELECT stvtrs.id,
                   stvtrs.parent,
                   stvtrs.askix
              FROM stvtrs
              JOIN _tree ON stvtrs.id = _tree.parent
             WHERE resid = new.resid
               AND stv_str_revnr IS NULL
          )

          SELECT askix
            INTO _parent_ask_index
            FROM _tree
           WHERE askix IS NOT NULL
           ORDER BY id DESC
           LIMIT 1
        ;

        -- globale Zuschläge (Faktor) der übergeordneten ASK
        _glob_zuschlag_faktor := null;

        _glob_zuschlag_faktor :=
              sum( o7zk_proz ) / 100
            FROM op7zko
            WHERE o7zk_ix = _parent_ask_index
              AND o7zk_proz IS NOT NULL
        ;

        _glob_zuschlag_faktor := 1 + coalesce( _glob_zuschlag_faktor, 0 );

        new.glob_zuschlag_faktor := nullif(_glob_zuschlag_faktor, 1);


        -- Selbstkosten mit globale Zuschlägen beaufschlagen.
        new.selbstko := new.selbstko * _glob_zuschlag_faktor;

    END IF;


    -- Verkaufspreis-Basis gesamt anhand Kalkulationsmenge
    IF vkp_faktor > 0 AND new.selbstko > 0 THEN
        -- Standard: Selbstkosten mit VKP-Faktor des Artikels
        new.vkpbas := new.selbstko * vkp_faktor;

    -- Fall völlig unklar und nicht dokumentiert
    ELSIF new.askix IS NOT NULL THEN

        new.vkpbas := new.selbstko;

    -- keine Selbstkosten übernommen
    ELSE
        -- wir nehmen die VKP-Basis, welche aus dem Artikel geholt wurde, diese ist noch in GME
        new.vkpbas := new.vkpbas * new.stm;

    END IF;


    -- Probleme abfangen, wenn kein Fertiungsartikel
    IF NOT art_is_fertigung THEN

        -- hat aber dennoch Stückliste? Set?
        IF EXISTS(SELECT true FROM stv WHERE st_zn = new.aknr) THEN

            -- wenn Preisquelle EPreis, dann Unterbauteile dazurechnen.
            -- In Selbstkosten des Endartikels sind diese natürlich mit integriert (Lagerwert des fertigen Bauteils)
            IF new.selbstko_source_table LIKE 'epreis%' THEN
                -- wir sind eine eingekauft Baugruppe ohne ASK,
                -- dann zählen wir unsere eigenen Kaufteilkosten dazu. #9493. Damit stueckl__calc_parent richtig rechnet
                new.askix := 0;

            END IF;

        END IF;


        -- Status vom übergeordneten Artikel in Auflösung ermitteln.
        SELECT
          askix,
          selbstko_source_table
        FROM stvtrs
        WHERE resid = new.resid
          AND id = new.parent
          AND stv_str_revnr IS NULL
        INTO r
        ;

        -- übergeordneter Artikel ist Kaufteil (keine ASK) aber Preisherkunft ist nicht Lieferant.
        -- Dann sind dort die Unterbauteile bereits enhtalten (ak_hest) und wir behandeln die Unterbauteile wie Rohmaterial
        IF     r.askix IS NULL
           AND r.selbstko_source_table NOT LIKE 'SET'
           AND r.selbstko_source_table NOT LIKE 'epreis%'
        THEN
            -- zurücksetzen, da zu Funktionsbeginn pauschal alles gesetzt wird! #9493
            new.romat := true;

        END IF;

    END IF;


    -- Log für Artikelpreis-Probleme
    PERFORM
      create_stvtrs_res_log(
          new.resid,
          new.aknr,
          'art_ekpreis',
          new.selbstko::varchar,
          new.selbstko_source_table
      )
    ;


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER stvtrs__b_i__calcprices
    BEFORE INSERT
    ON stvtrs
    FOR EACH ROW
    WHEN (NOT stvtrstriggerdisabled() AND new.kalku)
    EXECUTE PROCEDURE stvtrs__b_i__calcprices();
--

--
CREATE OR REPLACE FUNCTION stvtrs__a_d() RETURNS TRIGGER AS $$
  BEGIN
    DELETE FROM stvtrs_ksbedarf WHERE ksb_resid=old.resid;
    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER stvtrs__a_d
    AFTER DELETE
    ON stvtrs
    FOR EACH ROW
    EXECUTE PROCEDURE stvtrs__a_d();
--

-- Ermittelt und setzt die Kalkulationsergebnisse der Stücklistenauflösung für die Beistellung an Lieferanten durch
CREATE OR REPLACE FUNCTION tartikel.stvtrs_kalk__beistellung_an_lieferanten(
    _stvtrs         stvtrs,
    _per_preissuche boolean
  ) RETURNS stvtrs AS $$
  BEGIN

    IF NOT _per_preissuche THEN

        _stvtrs := tartikel.stvtrs_kalk__by__art_et( _stvtrs );

        -- Eigene Kosten hinzuzählen. Damit stueckl__calc_parent richtig rechnet
          -- siehe stvtrs__b_i__calcprices
        _stvtrs.askix := 0;

    END IF;

    IF _per_preissuche THEN
        _stvtrs := tartikel.stvtrs_kalk__by__search_ekpreis( _stvtrs );
    END IF;


    RETURN _stvtrs;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Ermittelt und setzt die Kalkulationsergebnisse der Stücklistenauflösung
  -- per Artikel-Einzelteil-Kosten.
CREATE OR REPLACE FUNCTION tartikel.stvtrs_kalk__by__art_et( _stvtrs stvtrs ) RETURNS stvtrs AS $$
  DECLARE
      _artikeldaten record;
  BEGIN

    -- Ohne Artikel keine Bearbeitung möglich
    IF _stvtrs.aknr IS NULL THEN
        RETURN _stvtrs;
    END IF;


    SELECT
      ak_hest_et,
      ak_vkpbas_et
    FROM art
    WHERE ak_nr = _stvtrs.aknr
    INTO
      _artikeldaten
    ;

    IF _artikeldaten IS NOT NULL THEN

        _stvtrs.selbstko  := _artikeldaten.ak_hest_et;
        _stvtrs.vkpbas    := _artikeldaten.ak_vkpbas_et;

        _stvtrs.selbstko_source_table := 'art';

    END IF;


    RETURN _stvtrs;
  END $$ LANGUAGE plpgsql STABLE;
--

-- Ermittelt und setzt die Kalkulationsergebnisse der Stücklistenauflösung
  -- per Preissuche.
CREATE OR REPLACE FUNCTION tartikel.stvtrs_kalk__by__search_ekpreis( _stvtrs stvtrs ) RETURNS stvtrs AS $$
  DECLARE
      _preissuche_ergebnis record;
  BEGIN

    -- Ohne Artikel keine Bearbeitung möglich
    IF _stvtrs.aknr IS NULL THEN
        RETURN _stvtrs;
    END IF;


    -- Alles Default (Rahmen, Epreise, Selbstko), außer roundToLos auf FALSE,
    -- [LG 09/2017] - von preis_uf1_basisw (Ohne Abzu u. Rabatt) auf preis_uf1_basisw_abzu geändert (mit Abzu u. Rabatt)
    SELECT
      preis_uf1_basisw_abzu,
      source_table
    FROM
        TWawi.Search_EKPreis(
            -- Artikel
            _stvtrs.aknr,
            -- Menge in GME
            tartikel.me__menge__in__menge_uf1(
                -- ME-ID ermitteln
                coalesce( _stvtrs.stmgc, tartikel.me__art__artmgc__m_id__by__ak_standard_mgc( _stvtrs.aknr ) ),
                -- Menge in ME
                _stvtrs.stm
            ),
            -- kein Fert.artikel, Lieferant = alle, ohne besondere Suchreihenfolge
            null,                 '%',              '',
            -- Preishint, Normprüfung, Losrundung
            false,        true,        false
        )
    INTO
      _preissuche_ergebnis
    ;

    IF _preissuche_ergebnis IS NOT NULL THEN

        _stvtrs.selbstko              := _preissuche_ergebnis.preis_uf1_basisw_abzu;
        _stvtrs.selbstko_source_table := _preissuche_ergebnis.source_table;

    END IF;


    RETURN _stvtrs;
  END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE TABLE stvtrs_ksbedarf (
  ksb_id        SERIAL PRIMARY KEY,
  ksb_resid     VARCHAR,
  ksb_faz       DATE,
  ksb_sez       DATE,
  ksb_ks        VARCHAR,
  ksb_zeit_stu      NUMERIC
 );
 --


--
CREATE TABLE stvtxt (
  sttxt_id      SERIAL NOT NULL PRIMARY KEY REFERENCES stv ON UPDATE CASCADE ON DELETE CASCADE,
  sttxt_st_id       INTEGER REFERENCES stv,
  sttxt_txt     TEXT
 );
 --

-- Optionsartikel
CREATE OR REPLACE FUNCTION artisoptart(_aknr varchar) RETURNS bool
     AS $$
        SELECT EXISTS(SELECT true FROM artoption_arts WHERE aoa_g_ak_nr = _aknr);
     $$ LANGUAGE sql STABLE PARALLEL SAFE;
--

-- 1. Mengeneinheiten  ... https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Art
 --https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Art
CREATE OR REPLACE FUNCTION tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(_aknr varchar) RETURNS integer  AS $$
      SELECT m_id
        FROM public.artmgc, public.art
       WHERE ak_nr = _aknr
         AND m_ak_nr = ak_nr
         AND m_mgcode = ak_standard_mgc;
      $$ LANGUAGE sql STABLE PARALLEL SAFE;

 CREATE OR REPLACE FUNCTION tartikel.me__art__artmgc__m_id__by__mgc(_aknr varchar, _mgc integer) RETURNS integer AS $$
      SELECT m_id
        FROM public.artmgc
       WHERE m_ak_nr = _aknr
         AND m_mgcode = _mgc;
      $$ LANGUAGE sql STABLE PARALLEL SAFE;
  --
  CREATE OR REPLACE FUNCTION z_99_deprecated.standard_mgc_id(_aknr varchar) RETURNS integer AS $$
      SELECT tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(_aknr);
      $$ LANGUAGE sql STABLE PARALLEL SAFE;
  --
 CREATE OR REPLACE FUNCTION art__standard_mgc_id(_aknr varchar) RETURNS integer AS $$ --alias
      SELECT tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(_aknr);
      $$ LANGUAGE sql STABLE PARALLEL SAFE;
 --
 CREATE OR REPLACE FUNCTION tartikel.me__art__artmgc__m_id__by__me_iso(_aknr varchar, _isoname varchar) RETURNS integer AS $$
      SELECT m_id
        FROM artmgc, art, mgcode
       WHERE m_ak_nr = _aknr
         AND ak_nr = m_ak_nr
         AND m_mgcode = me_cod
         AND me_iso = _isoname;
      $$ LANGUAGE sql STABLE PARALLEL SAFE;
  --
  CREATE OR REPLACE FUNCTION z_99_deprecated.getmgc_id_byiso(_aknr varchar, _isoname varchar) RETURNS integer AS $$
      SELECT tartikel.me__art__artmgc__m_id__by__me_iso(_aknr, _isoname);
      $$ LANGUAGE sql STABLE PARALLEL SAFE;
 --
 CREATE OR REPLACE FUNCTION tartikel.me__mgcode_iso(_mecod integer) RETURNS varchar AS $$
      SELECT coalesce(me_iso, me_bez) FROM mgcode WHERE me_cod=_mecod;
      $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;

  CREATE OR REPLACE FUNCTION tartikel.me_mgcode_iso(_mecod integer) RETURNS varchar AS $$
      SELECT tartikel.me__mgcode_iso(_mecod);
      $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;
 --
  CREATE OR REPLACE FUNCTION tartikel.me_code__from__me_iso( IN _me_iso varchar ) RETURNS integer AS $$
      SELECT me_cod FROM mgcode WHERE lower( me_iso ) = lower( _me_iso) OR lower( me_bez ) = lower( _me_iso) ;
      $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;
 ---
  CREATE OR REPLACE FUNCTION tartikel.me__x__m_uf__div(
        mid integer,
        x numeric
    ) RETURNS numeric(12,8) AS $$

        SELECT x / m_uf
          --   Hinweis: hier Public erforderlich, da Funktion bei DB-Erstellung aufgerufen wird
          FROM public.artmgc
         WHERE m_id = mid;

  $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;

  CREATE OR REPLACE FUNCTION tartikel.me__x__m_uf__mult(
        mid integer,
        x numeric
    ) RETURNS numeric(12,8) AS $$

        SELECT x * m_uf
          -- Hinweis: hier Public erforderlich, da Funktion bei DB-Erstellung aufgerufen wird
          FROM public.artmgc
         WHERE m_id = mid;

  $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;
 --
  CREATE OR REPLACE FUNCTION tartikel.me__menge__in__menge_uf1(
        mid integer,
        stk numeric
    ) RETURNS numeric AS $$

        -- Menge von EingabeME zu ME in UF1
        SELECT
          -- #14952 Absicherung freie Artikel
          coalesce(
            tartikel.me__x__m_uf__div( mid, stk ),
             stk
          );

   $$ LANGUAGE sql STABLE PARALLEL SAFE;
   --
   CREATE OR REPLACE FUNCTION Z_99_Deprecated.do_uf_1(artmgcid INTEGER, qty NUMERIC) RETURNS NUMERIC AS $$
     SELECT tartikel.me__menge__in__menge_uf1(artmgcid, qty);
    $$ LANGUAGE sql STABLE PARALLEL SAFE;
   --
 CREATE OR REPLACE FUNCTION me__menge__in__menge_uf1(mid INTEGER, stk NUMERIC) RETURNS NUMERIC AS $$ --Alias
    SELECT tartikel.me__menge__in__menge_uf1(mid, stk);
   $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;
 --
  CREATE OR REPLACE FUNCTION tartikel.me__menge_uf1__in__menge(
        mid integer,
        stk numeric
    ) RETURNS numeric AS $$

        -- Menge in UF1 nach EingabeME
        SELECT
          -- #14952 Absicherung freie Artikel
          coalesce(
              tartikel.me__x__m_uf__mult( mid,  stk ),
              stk
          );

   $$ LANGUAGE sql STABLE PARALLEL SAFE;
   --
   CREATE OR REPLACE FUNCTION Z_99_Deprecated.get_crowd_uf(mid INTEGER, stk NUMERIC) RETURNS NUMERIC AS $$
     SELECT tartikel.me__menge_uf1__in__menge(mid, stk);
    $$ LANGUAGE sql STABLE STRICT;
   --Rechnet IN Mengeneinheit artmgcID um, wenn Crowd in Grund-ME angegeben ist. get_crowd_uf( m_id, do_uf_1(m_id2, menge)) AS lg_anztot
 CREATE OR REPLACE FUNCTION me__menge_uf1__in__menge(mid INTEGER, stk NUMERIC) RETURNS NUMERIC AS $$ --Alias
    SELECT tartikel.me__menge_uf1__in__menge(mid, stk);
   $$ LANGUAGE sql STABLE STRICT;
 --
 CREATE OR REPLACE FUNCTION tartikel.me__preis_uf1__in__preis(mid INTEGER, price NUMERIC) RETURNS NUMERIC AS $$ -- Preis in UF1 nach EingabeME
     SELECT tartikel.me__x__m_uf__div(mid, price);
   $$ LANGUAGE sql STABLE STRICT;
   --
   CREATE OR REPLACE FUNCTION Z_99_Deprecated.do_waer_uf(mid INTEGER, price NUMERIC) RETURNS NUMERIC AS $$ --Preis von UF1 in EingabeME
     SELECT tartikel.me__preis_uf1__in__preis(mid, price);--Preis: Division
    $$ LANGUAGE sql STABLE STRICT;
   --
 CREATE OR REPLACE FUNCTION me__preis_uf1__in__preis(mid INTEGER, price NUMERIC) RETURNS NUMERIC AS $$ -- Alias
     SELECT tartikel.me__preis_uf1__in__preis(mid, price);
   $$ LANGUAGE sql STABLE STRICT;
 --
 CREATE OR REPLACE FUNCTION tartikel.me__preis__in__preis_uf1(mid INTEGER, price NUMERIC) RETURNS NUMERIC AS $$ --Preis von EingabeME zu ME in UF1
    SELECT tartikel.me__x__m_uf__mult(mid, price);
   $$ LANGUAGE sql STABLE STRICT;
   --
   CREATE OR REPLACE FUNCTION Z_99_Deprecated.do_waer_uf_1(artmgcid INTEGER, price NUMERIC) RETURNS NUMERIC AS $$ --Preis von EingabeME in UF1
     SELECT tartikel.me__preis__in__preis_uf1(artmgcid, price);
    $$ LANGUAGE sql STABLE STRICT;
   --
   CREATE OR REPLACE FUNCTION Z_99_Deprecated.do_waer_uf1(artmgcid INTEGER, price NUMERIC) RETURNS NUMERIC AS $$ --Preis von EingabeME in UF1
     SELECT tartikel.me__preis__in__preis_uf1(artmgcid, price);
    $$ LANGUAGE sql STABLE STRICT;
   --
 CREATE OR REPLACE FUNCTION me__price__to__price_uf1(mid INTEGER, price NUMERIC) RETURNS NUMERIC AS $$ --Alias
    SELECT tartikel.me__preis__in__preis_uf1(mid, price);
   $$ LANGUAGE sql STABLE STRICT;
 --
--
CREATE OR REPLACE FUNCTION ic_for_ac(IN _ac varchar) RETURNS integer AS $$
    SELECT ac_i FROM artcod WHERE ac_n = _ac;
    $$ LANGUAGE SQL STABLE PARALLEL SAFE;
--

--
CREATE OR REPLACE FUNCTION GetArtIc(IN _aknr varchar) RETURNS integer AS $$
    SELECT ac_i FROM artcod JOIN art ON ak_ac = ac_n AND ak_nr = _aknr;
    $$ LANGUAGE SQL STABLE PARALLEL SAFE;
--


-- Zuständige Mitarbeiter für einen Artikel
CREATE TABLE artZust (
  az_id             SERIAL PRIMARY KEY,
  az_aknr   VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,    -- Artikel um den es geht
  az_ll_db_usename  VARCHAR(20) NOT NULL -- Mitarbeiternummer der Artikel betreut
  );
 --

 CREATE UNIQUE INDEX artZust_aknr_llminr ON artZust(az_aknr, az_ll_db_usename);

 CREATE OR REPLACE FUNCTION artzust__ismine(aknr VARCHAR) RETURNS VARCHAR(100) AS $$--gibt "current_user" zurück wenn der aktuell angemeldete Mitarbeiter für einen Artikel zuständig ist, sonst 0.
  DECLARE iszust BOOLEAN;
  BEGIN
   SELECT true INTO iszust FROM artzust WHERE az_aknr=aknr AND az_ll_db_usename=current_user;
   IF COALESCE(iszust, FALSE) THEN
    RETURN 'mine';
   ELSE
    RETURN NULL;
   END IF;
  END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE TABLE ArtAenderung (
  aa_nr         VARCHAR(20) PRIMARY KEY,        -- Änderungsnummer aus Nummernkreis
  aa_aknr   VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,    -- Für Artikel
  aa_bez        VARCHAR(50),                -- Änderungsbezeichnung
  aa_txt        TEXT,                   -- Zusatztext
  aa_txt_rtf    TEXT,
  aa_beginn DATE,                   -- Die Änderung begann/beginnt an dem Tag
  aa_ende   DATE,                   -- Alle Änderungen wurden bis dann durchgeführt
  aa_abt1   BOOLEAN NOT NULL DEFAULT FALSE,     -- Flags ob Abteilungen Änderung durchgeführt haben
  aa_abt2   BOOLEAN NOT NULL DEFAULT FALSE,
  aa_abt3   BOOLEAN NOT NULL DEFAULT FALSE,
  aa_abt4   BOOLEAN NOT NULL DEFAULT FALSE
 );
--


-- Artikelstatus und zugehörige Farbcodierung ausgeben, holt Artikelstatus-Daten per Subselect auf art

 CREATE OR REPLACE FUNCTION tartikel.art__artikelstatus__by__ak_nr
                                            (IN aknr VARCHAR,
                                             OUT artstatus VARCHAR(30), --N, G, AU,UN (Neu, Güig, Auslauf, Ungültig)
                                             OUT artvalidGNAU BOOLEAN,  --N+G+AU
                                             OUT artvalidG BOOLEAN,     --Nur G
                                             OUT artUngueltig BOOLEAN,
                                             OUT artstatus_cimgreen BOOLEAN, OUT artstatus_cimyellow BOOLEAN, OUT artstatus_cimred BOOLEAN
                                             ) RETURNS RECORD AS $$
     DECLARE
       r RECORD;
     BEGIN
       SELECT (tartikel.art__artikelstatus__by__flags(ak_neuanlage, ak_auslauf, ak_Auslauf_Aufbrauch)).* INTO r FROM public.art WHERE ak_nr = aknr;
       artStatus    := r.ArtStatus;
       artvalidG    := r.ArtStatus='G';
       artvalidGNAU := artvalidG OR r.ArtStatus IN ('N', 'AU');
       artstatus_cimgreen     := r.artstatus_CimGreen;
       artstatus_cimyellow    := r.artstatus_CimYellow;
       artstatus_cimred       := r.artstatus_CimRed;
       artUngueltig := r.artstatus_cimred;
      END $$ LANGUAGE plpgsql STABLE STRICT;
 --
 -- Version, die mit den reingegebenen Werten arbeitet und ohne SubSelect auskommt.
 CREATE OR REPLACE FUNCTION tartikel.art__artikelstatus__by__flags
                                            (IN akNeuanlage BOOLEAN, IN akAuslauf DATE, IN akAuslaufAufbrauch BOOLEAN,
                                             OUT artstatus VARCHAR(30),
                                             OUT artstatus_cimgreen BOOLEAN,
                                             OUT artstatus_cimyellow BOOLEAN,
                                             OUT artstatus_cimred BOOLEAN
                                             ) RETURNS RECORD AS $$
  DECLARE neu       BOOLEAN;
          aenderung BOOLEAN;
          auslauf   BOOLEAN;
          ungueltig BOOLEAN;
          gueltig   BOOLEAN;

  BEGIN

       Artstatus:='';

       -- Statusflags bestimmen, derzeit 4 Stati vergeben
       neu                 := akNeuanlage;
       aenderung           := false;  -- Änderungskennzeichen wurde nie berücksichtigt?

       auslauf             :=  ((akAuslauf IS NOT NULL) AND (akAuslauf  > current_date)) OR akAuslaufAufbrauch;
       ungueltig           :=   (akAuslauf IS NOT NULL) AND (akAuslauf <= current_date)  AND NOT akAuslaufAufbrauch;
       gueltig             := NOT(neu OR aenderung OR auslauf OR ungueltig);

       -- Farbcodierung je nach Status wählen
       artstatus_cimgreen  := neu;
       artstatus_cimyellow := aenderung OR auslauf;
       artstatus_cimred    := ungueltig;

       artstatus           := Trim(TSystem.IfThen(gueltig,'G ','') || TSystem.IfThen(neu,'N ','') || TSystem.IfThen(aenderung,'AE ','') || TSystem.IfThen(auslauf,'AU ','') || TSystem.IfThen(ungueltig,'UN ',''));

       RETURN;
  END $$ LANGUAGE plpgsql STABLE;
 --


 CREATE OR REPLACE FUNCTION z_99_deprecated.GetArtikelStatus(IN _for_aknr VARCHAR,
                                             OUT artstatus VARCHAR(30), --N, G, AU,UN (Neu, Güig, Auslauf, Ungültig)
                                             OUT artvalidGNAU BOOLEAN,  --N+G+AU
                                             OUT artvalidG BOOLEAN,     --Nur G
                                             OUT artUngueltig BOOLEAN,
                                             OUT cimgreen BOOLEAN, OUT cimyellow BOOLEAN, OUT cimred BOOLEAN
                                             ) RETURNS RECORD AS $$
     SELECT artstatus, artvalidGNAU, artvalidG, artUngueltig, artstatus_cimgreen AS cimgreen, artstatus_cimyellow AS cimyellow, artstatus_cimred AS cimred
            FROM tartikel.art__artikelstatus__by__ak_nr(_for_aknr);
  $$ LANGUAGE SQL STABLE STRICT;
 --
 CREATE OR REPLACE FUNCTION z_99_deprecated.GetArtikelStatus(IN _in_akNeuanlage BOOLEAN, IN _in_akAuslauf DATE, IN _in_akAuslaufAufbrauch BOOLEAN,
                                             OUT artstatus VARCHAR(30),
                                             OUT cimgreen BOOLEAN,
                                             OUT cimyellow BOOLEAN,
                                             OUT cimred BOOLEAN
                                             ) RETURNS RECORD AS $$

     SELECT artstatus, artstatus_cimgreen AS cimgreen, artstatus_cimyellow AS cimyellow, artstatus_cimred AS cimred
            FROM tartikel.art__artikelstatus__by__flags(_in_akNeuanlage, _in_akAuslauf, _in_akAuslaufAufbrauch);
  $$ LANGUAGE SQL STABLE;
--

/* Haupttabelle Werkzeugverwaltung, N->1 Zuordnung von Werkzeugen, zu Artikeln

Anmerkung LG: Wenn das mal umgebaut wird, sollte man das wz_lagernd raushauen. Das ist immer abhaengig von lpdat und
Ausgaben, sollte also nur berechnet sein und nicht hier gespeichert.*/
CREATE TABLE werkzeug (
  wz_id           SERIAL PRIMARY KEY,
  wz_aknr       VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,--Artikelnummer des Werkzeugs
  wz_wznr       VARCHAR(50) UNIQUE,                         --Werkzeug- /Inventar-Nr.
  wz_hest       VARCHAR(50),                --Hersteller
  wz_ibdat      DATE,                   --Datum Inbetriebnahme
  wz_lpdat      DATE,                   --Datum Aussonderung
  wz_lagernd        BOOL NOT NULL DEFAULT TRUE,     --Werkzeug lagernd
  wz_txt        TEXT,                   --Freier Zusatztext
  wz_txt_rtf        TEXT,
  wz_sernummer      VARCHAR(50),                --Geraetenummer (Herstellerseitig, quasi Werkzeugseriennummer)
  wz_leasedat       DATE,                   --Ablauf Leasingdatum. Da muss es wieder da sein.
  wz_allg1      VARCHAR(80),                --Allgmein Feld (bei SunStrom z.B. Verwaltungsort)
  wz_allgb1     BOOLEAN NOT NULL DEFAULT false      --Allgemeinfeld, zB Überwachen oder nicht
 /*  wz_ausort      VARCHAR(50),                --Ausgabeort
  wz_ausdat     DATE,                   --Datum Ausgabe
  wz_rueckdat       DATE,                   --Datum Rücklieferung
  wz_ausvon     VARCHAR(10),                --Mitarbeiternummer des ausgebenden Mitarbeiters
  wz_ausan      VARCHAR(10),                --Mitarbeiternummer des empfangenden Mitarbeiters*/
 );
 --

 --CREATE UNIQUE INDEX wz_aknr_wznr ON werkzeug (wz_aknr,wz_wznr);

 --
 CREATE OR REPLACE FUNCTION werkzeug__b_iu() RETURNS TRIGGER AS $$
    BEGIN
     IF new.wz_lpdat IS NOT NULL THEN--aussonderung
          new.wz_lagernd:=FALSE;
     ELSE
          new.wz_lagernd:=NOT EXISTS(SELECT true FROM werkzeugausg WHERE new.wz_id=awz_wz_id AND awz_rueckdat IS NULL);--nicht ausgegeben
     END IF;
     --
     RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER werkzeug__b_iu
     BEFORE INSERT OR UPDATE
     ON werkzeug
     FOR EACH ROW
     EXECUTE PROCEDURE werkzeug__b_iu();
--

-- Subtabelle für Werkzeugverwaltung-Vorgaben
CREATE TABLE wz_presets (
  wzp_allg1     VARCHAR(80) -- fuer werkzeug.wz_allg1
 );
--

-- Subtabelle zum Verlauf der Ausgbe der Werkzeuge
CREATE TABLE werkzeugausg (
  awz_id        SERIAL PRIMARY KEY,
  awz_wz_id     INTEGER NOT NULL REFERENCES werkzeug ON UPDATE CASCADE,
  awz_ausort        VARCHAR(50),                --Ausgabeort
  awz_ausdat        DATE NOT NULL DEFAULT current_date, --Datum Ausgabe
  awz_vrueckdat     DATE,                   --voraussichtliches Rückgabedatum
  awz_rueckdat      DATE,                   --Datum Rücklieferung
  awz_ausvon        VARCHAR(30) DEFAULT tsystem.current_user_ll_db_usename(),   --ausgebender Mitarbeiter
  awz_ausan     VARCHAR(10),                --Mitarbeiternummer des empfangenden Mitarbeiters
  awz_txt       TEXT,                   --Freier Zusatztext
  awz_txt_rtf       TEXT,
  awz_an_nr     VARCHAR(50)             --Projekt-Nr.
 );
 --

 --
 CREATE OR REPLACE FUNCTION werkzeugausg__a_iu() RETURNS TRIGGER AS $$
  BEGIN
   IF new.awz_rueckdat IS NULL AND EXISTS(SELECT true FROM werkzeugausg WHERE awz_rueckdat IS NULL AND awz_wz_id=new.awz_wz_id AND awz_id<>new.awz_id) THEN
    RAISE EXCEPTION '%', lang_text(11221);--Werkzeug kann nur einmal ausgegeben werden
   END IF;
   UPDATE werkzeug SET wz_id=wz_id WHERE wz_id=new.awz_wz_id;--aktualisieren Lagernd-Status
   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER werkzeugausg__a_iu
   AFTER INSERT OR UPDATE
   ON werkzeugausg
   FOR EACH ROW
   EXECUTE PROCEDURE werkzeugausg__a_iu();
 --

 --
 CREATE OR REPLACE FUNCTION werkzeugausg__a_d() RETURNS TRIGGER AS $$
  BEGIN
    --Wenn wir Ausleihen loeschen, setzen wir das Werkzeug wieder auf lagernd.
    IF NOT EXISTS(SELECT true FROM werkzeugausg WHERE awz_rueckdat IS NULL AND awz_wz_id=old.awz_wz_id ) THEN
        UPDATE werkzeug SET wz_lagernd=TRUE WHERE NOT wz_lagernd AND wz_id=old.awz_wz_id;
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER werkzeugausg__a_d
   AFTER DELETE
   ON werkzeugausg
   FOR EACH ROW
   EXECUTE PROCEDURE werkzeugausg__a_d();
 --

-- Definition Vorgaben für Kontrollmerkmale für Artikellagerbewegungen
CREATE TABLE artPruefungVorgabe ( --Ehemals ArtlagTxt
  aprv_id       SERIAL PRIMARY KEY,
  aprv_bez      VARCHAR(100) NOT NULL,                                                       --Bezeichnung oder Gruppe des Merkmals
  aprv_txt      Text,                                                                        --Beschreibungstext
  aprv_eing     BOOLEAN NOT NULL DEFAULT FALSE,                                              --Artikel muss bei Wareneingang kontrolliert werden => wird bei Nichtbestehen gesperrt
  aprv_ausg     BOOLEAN NOT NULL DEFAULT FALSE,                                              --Artikel wird bei Lagerabgang geprüft => bei Nichtbestehen Lagerabgang abgebrochen
  aprv_web      BOOLEAN NOT NULL DEFAULT FALSE,                                              --Wareneingangsbericht benötigt
  aprv_aufdok   BOOLEAN NOT NULL DEFAULT FALSE,                                              --automatisch auf Dokumenten anzeigen
  aprv_doktxt   Text,                                                                        --Dokumententext (wenn aprv_aufdok=TRUE)
  aprv_abz_id   INTEGER REFERENCES abzu ON UPDATE CASCADE                  --Ab-/Zuschlags-ID; ohne Referenz, weil Kann - nicht Muss
  -- aprv_ausw      BOOLEAN NOT NULL DEFAULT FALSE  Entfällt, da abgebildet über apr_web und apr_eing
 );
--

-- Standardvorgaben für einen Artikel im Artikelstamm
CREATE TABLE artPruefung (
  apr_id         SERIAL PRIMARY KEY,
  apr_pos        SMALLINT NOT NULL,                                                          --Position/Reihenfolge
  apr_aknr       VARCHAR(40) REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,             --Zugeordneter Artikel
  apr_ac         VARCHAR(9) REFERENCES artcod ON UPDATE CASCADE ON DELETE CASCADE,           --alternative zugeordneter über AC
  apr_bez        VARCHAR(100) NOT NULL,                                                      --Bezeichnung der Artikelprüfung
  apr_txt        TEXT,                                                                       --Ergänzender Zusatztext
  apr_eing       BOOL NOT NULL DEFAULT FALSE,                                                --Prüfung erfolgt beim Lagerzugang
  apr_ausg       BOOL NOT NULL DEFAULT FALSE,                                                --Prüfung erfolgt beim Lagerabgang
  apr_web        BOOL NOT NULL DEFAULT FALSE,                                                --Prüfung erfolgt im WEB (Wareneingangsbericht)
  apr_aufdok     BOOL NOT NULL DEFAULT FALSE,                                                --automatisch auf Dokumenten anzeigen
  apr_doktxt     Text,                                                                       --Dokumententext (wenn aprv_aufdok=TRUE)
  apr_abz_id     INTEGER REFERENCES abzu ON UPDATE CASCADE,                --Ab-/Zuschlags-ID; ohne Referenz, weil Kann - nicht Muss
  apr_apnr       VARCHAR(40) REFERENCES art ON UPDATE CASCADE,             --Arbeitspaket, im Falle einer Auswärtsbearbeitung
  CONSTRAINT     artPruefung_reference_valid CHECK (apr_aknr IS NOT NULL OR apr_ac IS NOT NULL)
 );
 --

 CREATE UNIQUE INDEX artpruefung_aknr_pos ON artpruefung(apr_aknr, apr_pos);
 CREATE INDEX artPruefung_apr_ac ON artPruefung(apr_ac);

 --
 CREATE OR REPLACE FUNCTION artpruefung__b_id_checkUserRights() RETURNS TRIGGER AS $$
  DECLARE web BOOLEAN;
  BEGIN
   --Nur SYS.QAB-Admins dürfen WEB-Pruefungen löschen
   IF (tg_op='DELETE') THEN
    IF (old.apr_web) AND (current_user <> 'syncro') AND (NOT current_user IN (
            SELECT usename FROM pg_group, pg_user WHERE usesysid=ANY(grolist) AND groname='SYS.QAB-Admin'))
        THEN
      RAISE EXCEPTION 'xtt10560';
    END IF;
    RETURN old;
   END IF;
   --Einfügen von SYS.WEB-Merkmalen nur durch SYS.QAB-Admin oder WEB-Pruefer
   IF (tg_op = 'INSERT') THEN
    IF new.apr_web AND (current_user <> 'syncro') AND
        NOT ((current_user IN (SELECT usename FROM pg_group, pg_user WHERE usesysid=ANY(grolist) AND groname='SYS.QAB-Admin'))
         OR  (current_user IN (SELECT usename FROM pg_group, pg_user WHERE usesysid=ANY(grolist) AND groname='SYS.WEB-Pruefer')))
    THEN
          RAISE EXCEPTION 'xtt10561';
    END IF;

    RETURN new;
   END IF;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artpruefung__b_id_checkUserRights
   BEFORE INSERT OR DELETE
   ON artpruefung
   FOR EACH ROW
   EXECUTE PROCEDURE artpruefung__b_id_checkUserRights();
--
  CREATE OR REPLACE FUNCTION artPruefung__b_d() RETURNS TRIGGER AS $$
  BEGIN
    -- #11165 - Löschen ist nicht zulässig, weil dann die Vorgabe aus den AC's für alle Artikel verschwindet.
    -- Im Einzelfall muß vorerst der Support den Trigger manuell deaktivieren, löschen und wieder aktivieren.

    RAISE EXCEPTION 'xtt30188'; -- Artikelprüfungen eines AC's dürfen nicht gelöscht werden!

    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artPruefung__b_d
    BEFORE DELETE
    ON artPruefung
    FOR EACH ROW
    WHEN (old.apr_ac IS NOT NULL) --nur Ausführen, ein AC eine Artikelprüfung vorgibt
    EXECUTE PROCEDURE artPruefung__b_d();
--

--Erzeugt für einen Artikel aus den Vorgaben ein Kontrollmerkmal, gibt ID zurück.
CREATE OR REPLACE FUNCTION CreateArtPruefung(aprvid INTEGER, apraknr VARCHAR(40)) RETURNS INTEGER AS $$
  DECLARE aprid  INTEGER;
          pos INTEGER;
  BEGIN
    pos:= COALESCE((SELECT MAX(apr_pos) FROM artpruefung WHERE apr_aknr = apraknr), 0) + 1;
    INSERT INTO artpruefung (apr_pos, apr_aknr, apr_bez, apr_txt, apr_eing, apr_ausg, apr_web, apr_aufdok, apr_doktxt, apr_abz_id)
        SELECT pos, apraknr, aprv_bez, aprv_txt, aprv_eing, aprv_ausg, aprv_web, aprv_aufdok, aprv_doktxt, aprv_abz_id FROM artpruefungvorgabe WHERE aprv_id = aprvid
    RETURNING apr_id INTO aprid;
    RETURN aprid;
  END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

--Tabelle für die Tests, die für bestimmte Vorgänge mit einem Artikel durchgeführt werden sollen bzw. wurden.
CREATE TABLE artPruefungTest (
  aprt_id               SERIAL PRIMARY KEY,
  aprt_pos              SMALLINT NOT NULL,                      --Position/Reihenfolge
  aprt_apr_id           INTEGER REFERENCES artpruefung ON UPDATE CASCADE ON DELETE SET NULL,
  aprt_bez              VARCHAR(100) NOT NULL,
  aprt_txt              TEXT,--Freier Ergänzungstext
  aprt_passed           BOOLEAN NOT NULL DEFAULT FALSE,                 --Kontrolle wurde bestanden
  aprt_failed           BOOLEAN NOT NULL DEFAULT FALSE,                 --Kontrolle wurde fehlgeschlagen
  aprt_pruefdat         TIMESTAMP(0) WITHOUT TIME ZONE,                 --Wann Kontrolle durchgeführt wurde.
  aprt_prueftxt         TEXT,                               --Kommentar warum evtl. nicht bestanden.
  aprt_ldid             INTEGER,  --REFERENCES ldsdok  in TableConstraints.Sql      --Pruefung fuer Bestellung
  aprt_wwen             INTEGER,  --REFERENCES wendat  in TableConstraints.Sql      --Pruefung fuer freien Wareneingang
  aprt_agid             INTEGER,  --REFERENCES auftg   in TableConstraints.Sql      --Pruefung fuer Auftrag
  aprt_lnr              INTEGER,  --REFERENCES lifsch  in TableConstraints.Sql      --Pruefung fuer freien Lagerabgang
  aprt_weknr            INTEGER,  --REFERENCES wareneingangskontrolle in TableConstraints.Sql --Pruefung vor Abschluss eines Wareneingangsberichts
  aprt_apnr             VARCHAR(40) REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE  --Bei Ruecklieferung auf dieses Auswärtspaket soll geprüft werden
  --aprt_aw_id(_obsolet)        INTEGER,  --REFERENCES auswlog  in TableConstraints.SQl
  --aprt_awr_id(_obsolet)       INTEGER   --REFERENCES auswlogrueck in TableConstraints.Sql --Pruefung vor Abschluss eines Wareneingangsberichts
 );
 --

 --
 CREATE OR REPLACE FUNCTION artPruefungTest__b_iu() RETURNS TRIGGER AS $$
  DECLARE r RECORD;
  BEGIN
   new.aprt_pruefdat:=COALESCE(new.aprt_pruefdat, current_Date);
   IF (new.aprt_apr_id = 0 ) THEN
     new.aprt_apr_id = NULL;
   END IF;
   RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER artPruefungTest__b_iu
    BEFORE INSERT OR UPDATE
    ON artPruefungTest
   FOR EACH ROW
   EXECUTE PROCEDURE artPruefungTest__b_iu();
  --

 /*Aufrufkonventionen
  ForTable = ldsdok             => ForID = ld_id,   FromID = null
  ForTable = wendat             => ForID = w_wen,   FromID = ld_id
  ForTable = auftg              => ForID = ag_id,   FromID = null
  ForTable = lifsch             => ForID = l_nr,    FromID = ag_id
  ForTable = wareneingangskontrolle     => ForID = wek_nr,  FromID =null
 */

--
CREATE OR REPLACE FUNCTION CopyArtTests(ForTable VARCHAR, ForID INTEGER, FromID INTEGER) RETURNS INTEGER AS $$
 DECLARE rows INTEGER;
 BEGIN
  IF (ForTable IS NULL) OR (ForID IS NULL) THEN
    RAISE EXCEPTION '%', Format(lang_text(29167) /*'Ungültige Parameter für CopyArtTests. Parameter ForTable und ForID müssen gefüllt sein.'*/);
  END IF;

  CASE lower(ForTable)
        WHEN 'ldsdok' THEN -- Artikelpruefungen zur Bearbeitung/Ergaenzung fuer Bestellungen aus Artikelstammpruefungen
            INSERT INTO artpruefungTest(aprt_pos, aprt_apr_id, aprt_bez, aprt_txt, aprt_ldid)
            SELECT row_number() OVER () + COALESCE((SELECT MAX(aprt_pos) FROM artPruefungTest WHERE aprt_ldid = ForID),0),
                apr_id, apr_bez, apr_txt, ForID
            FROM artpruefung
            JOIN ldsdok ON ld_id = ForID
            WHERE (apr_aknr = ld_aknr -- Artikelnummer
                OR apr_ac = ( SELECT ak_ac FROM art WHERE ak_nr = ld_aknr ) ) -- Artikelcode
               AND apr_eing -- nur WE Prüfungen
               AND apr_id NOT IN ( SELECT aprt_apr_id FROM artpruefungTest WHERE aprt_ldid = ForID AND aprt_apr_id IS NOT NULL ) -- doppeltes Einfügen ausschließen
            ORDER BY apr_pos;
        GET DIAGNOSTICS rows = ROW_COUNT;

        WHEN 'wendat' THEN --Artikelpruefungen zum Abfragen beim Wareneingang
            --Aus Bestellpruefungen kopiert
            INSERT INTO artpruefungTest(aprt_pos, aprt_apr_id, aprt_bez, aprt_txt, aprt_wwen)
            SELECT row_number() OVER () + COALESCE((SELECT MAX(aprt_pos) FROM artPruefungTest WHERE aprt_wwen = ForID),0),
                aprt_apr_id, aprt_bez, aprt_txt, ForID
            FROM artPruefungTest
            WHERE aprt_ldid = FromID
              AND apr_id NOT IN ( SELECT aprt_apr_id FROM artpruefungTest WHERE aprt_wwen = ForID AND aprt_apr_id IS NOT NULL ) -- doppeltes Einfügen ausschließen
            ORDER BY aprt_pos;
        GET DIAGNOSTICS rows = ROW_COUNT;

      --Artikelpruefungen zur Bearbeitung/Ergaenzung fuer Auftrag aus Artikelstammpruefungen
      WHEN 'auftg' THEN
            INSERT INTO artpruefungTest(aprt_pos, aprt_apr_id, aprt_bez, aprt_txt, aprt_agid)
              SELECT row_number() OVER () + COALESCE((SELECT MAX(aprt_pos) FROM artPruefungTest WHERE aprt_agid = ForID),0),
                apr_id, apr_bez, apr_txt,agid
            FROM artpruefung
            JOIN auftg ON  ag_id = ForID
            WHERE ( apr_aknr = ag_aknr -- Artikelnummer
                OR apr_ac = ( SELECT ak_ac FROM art WHERE ak_nr = ag_aknr ) ) -- Artikelcode
              AND apr_ausg
              AND apr_id NOT IN ( SELECT aprt_apr_id FROM artpruefungTest WHERE aprt_agid = ForID AND aprt_apr_id IS NOT NULL ) -- doppeltes Einfügen ausschließen
            ORDER BY apr_pos;
        GET DIAGNOSTICS rows = ROW_COUNT;

        WHEN 'lifsch' THEN --Artikelpruefungen zum Abfragen beim Lagerabgang
            --Aus Auftrag kopiert
            INSERT INTO artpruefungTest(aprt_pos, aprt_apr_id, aprt_bez, aprt_txt, aprt_lnr)
            SELECT row_number() OVER () + COALESCE((SELECT MAX(aprt_pos) FROM artPruefungTest WHERE aprt_lnr = ForID),0),
                aprt_apr_id, aprt_bez, aprt_txt, ForID
            FROM artPruefungTest
            WHERE aprt_agid = FromID
              AND apr_id NOT IN ( SELECT aprt_apr_id FROM artpruefungTest WHERE aprt_lnr = ForID AND aprt_apr_id IS NOT NULL ) -- doppeltes Einfügen ausschließen
            ORDER BY aprt_pos;
        GET DIAGNOSTICS rows = ROW_COUNT;

      WHEN 'wareneingangskontrolle' THEN
            INSERT INTO artpruefungTest(aprt_pos, aprt_apr_id, aprt_bez, aprt_txt, aprt_weknr)
            SELECT row_number() OVER () + COALESCE((SELECT MAX(aprt_pos) FROM artpruefungTest WHERE aprt_weknr = ForID),0),
                apr_id, apr_bez, apr_txt, ForID
            FROM artpruefung
            JOIN wareneingangskontrolle ON wek_nr = ForID
            WHERE ( apr_aknr = wek_ak_nr -- Artikelnummer
                 OR apr_ac = ( SELECT ak_ac FROM art WHERE ak_nr = wek_ak_nr ) ) -- Artikelcode
              AND apr_web
              AND apr_id NOT IN ( SELECT aprt_apr_id FROM artpruefungTest WHERE aprt_weknr = ForID AND aprt_apr_id IS NOT NULL ) -- doppeltes Einfügen ausschließen
            ORDER BY apr_pos;
        GET DIAGNOSTICS rows = ROW_COUNT;

      ELSE
        RAISE EXCEPTION '%', Format(lang_text(29168) /*'Ungültige Zieltabelle für CopyArtTests: %'*/, ForTable);
  END CASE;
  RETURN rows;
 END $$ LANGUAGE plpgsql VOLATILE;
--

--- #7715 Zusatztext Artikel (Langtexte Auftrag, Lieferschein, Rechnung) Mehrsprachig
CREATE TABLE artdokutxt
 (adtx_id               SERIAL PRIMARY KEY,
  adtx_ak_nr             VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE ON DELETE CASCADE,
  adtx_spr_key           VARCHAR(5)  NOT NULL REFERENCES sprach,
  adtx_auftxt           TEXT,
  adtx_auftxt_rtf       TEXT,
  adtx_bestxt           TEXT,
  adtx_bestxt_rtf       TEXT,
  adtx_lieftxt          TEXT,
  adtx_lieftxt_rtf      TEXT,
  adtx_faktxt            TEXT,
  adtx_faktxt_rtf       TEXT
 );

CREATE UNIQUE INDEX xtt29082 ON artdokutxt(adtx_ak_nr, adtx_spr_key);     -- 'Artikeltexte - Sprache'

--- #8475
CREATE OR REPLACE FUNCTION artdokutxt__a_u_adtx_auftxt() RETURNS trigger AS $$
  BEGIN
    UPDATE auftg SET ag_txt = new.adtx_auftxt, ag_txt_rtf = new.adtx_auftxt_rtf
    WHERE ag_astat <> 'I'
      AND ag_aknr LIKE new.adtx_ak_nr
      AND ag_dokunr IS NULL
      AND ag_stkl = 0
      AND NOT ag_done
      AND NOT ag_storno
      AND TRIM(ag_txt) LIKE TRIM(old.adtx_auftxt)
    ;
    IF old.adtx_auftxt IS DISTINCT FROM new.adtx_auftxt THEN
       PERFORM PRODAT_HINT('Auftragsdokumenttext:' || E'\r\n'
      || COALESCE(old.adtx_auftxt, '') || E'\r\n'
      || '           ist auf' || E'\r\n'
      || COALESCE(new.adtx_auftxt, '') || E'\r\n'
      || 'geändert.' || E'\r\n' || '   ');
    END IF;
    RETURN new;
 END $$ LANGUAGE plpgsql VOLATILE;

 CREATE TRIGGER artdokutxt__a_u_adtx_auftxt
  AFTER UPDATE OF adtx_auftxt
  ON artdokutxt
  FOR EACH ROW
  WHEN ((new.adtx_spr_key = prodat_languages.curr_lang())AND((new.adtx_auftxt IS DISTINCT FROM old.adtx_auftxt)OR(new.adtx_auftxt_rtf IS DISTINCT FROM old.adtx_auftxt_rtf)))
  EXECUTE PROCEDURE artdokutxt__a_u_adtx_auftxt();
---

--Trigger an Art
CREATE OR REPLACE FUNCTION art__a_iu_artdokutxt() RETURNS TRIGGER AS $$
  BEGIN
    IF tg_op='INSERT' THEN --adtx
      IF NOT EXISTS (SELECT true FROM artdokutxt WHERE adtx_ak_nr = new.ak_nr AND adtx_spr_key = prodat_languages.curr_lang()) THEN
        INSERT INTO artdokutxt (adtx_ak_nr, adtx_spr_key, adtx_auftxt, adtx_bestxt, adtx_lieftxt, adtx_faktxt, adtx_auftxt_rtf, adtx_bestxt_rtf, adtx_lieftxt_rtf, adtx_faktxt_rtf)
         VALUES (new.ak_nr, prodat_languages.curr_lang(), new.ak_auftxt, new.ak_bestxt, new.ak_lieftxt, new.ak_faktxt, new.ak_auftxt_rtf, new.ak_bestxt_rtf, new.ak_lieftxt_rtf, new.ak_faktxt_rtf);
      END IF;
    ELSE
        UPDATE artdokutxt SET
          adtx_auftxt = new.ak_auftxt, adtx_bestxt = new.ak_bestxt, adtx_lieftxt = new.ak_lieftxt, adtx_faktxt = new.ak_faktxt,
          adtx_auftxt_rtf = new.ak_auftxt_rtf, adtx_bestxt_rtf = new.ak_bestxt_rtf, adtx_lieftxt_rtf = new.ak_lieftxt_rtf, adtx_faktxt_rtf = new.ak_faktxt_rtf
         WHERE adtx_ak_nr = new.ak_nr AND adtx_spr_key = prodat_languages.curr_lang();
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

 CREATE TRIGGER art__a_iu_artdokutxt
    AFTER INSERT OR UPDATE OF ak_auftxt, ak_bestxt, ak_lieftxt, ak_faktxt, ak_auftxt_rtf, ak_bestxt_rtf, ak_lieftxt_rtf, ak_faktxt_rtf
    ON art
    FOR EACH ROW
    EXECUTE PROCEDURE art__a_iu_artdokutxt();

--- Funktionen zum Holen der Sprachtexte
CREATE OR REPLACE FUNCTION tartikel.adtx_getArtTxtLang(IN _ak_nr VARCHAR, IN _ident VARCHAR, IN _krz VARCHAR DEFAULT NULL, IN _lang VARCHAR DEFAULT NULL, OUT txt TEXT, OUT txtrtf TEXT) AS $$
 --- Engangsparameter  : _lang   - Sprache,
 ---                     _ak_nr  - Artikel-Nr,
 ---                     _ident  - Identifikation: 'AA' - Angebot/Auftrag, 'EK' - Enkauf, 'LS' - Lieferschein, 'RE' - Rechnung,
 ---                     _getrtf - RTF-Text oder Text zurückgeben
 --- Rückgabewert      : Artikeltext oder Artikel-RTF_Text
    DECLARE intlang VARCHAR;
 BEGIN
   intlang:=_lang;
   IF (intlang IS NULL OR intlang = '') AND _krz IS NOT NULL THEN
      CASE _ident
         WHEN 'AA'  THEN intlang := prodat_languages.customerlang(_krz);  -- Angebot/Auftrag
         WHEN 'LS'   THEN intlang := prodat_languages.customerlang(_krz);  -- Lieferschein
         WHEN 'RE'   THEN intlang := prodat_languages.customerlang(_krz);  -- Rechnung
         --
         WHEN 'EK'   THEN intlang := prodat_languages.supplierlang(_krz);  -- Einkauf
      END CASE;
   END IF;
   IF intlang IS NULL OR intlang = '' THEN
     intlang := prodat_languages.curr_lang();
   END IF;
   IF _ident = 'AA' THEN
     SELECT adtx_auftxt, adtx_auftxt_rtf INTO txt, txtrtf FROM artdokutxt WHERE adtx_ak_nr = _ak_nr AND adtx_spr_key = UPPER(intlang);
     IF NOT FOUND OR txt IS NULL THEN -- in abweichender Sprache nix gefunden, dann Artikeltext
       SELECT ak_auftxt, ak_auftxt_rtf INTO txt, txtrtf FROM art WHERE ak_nr = _ak_nr;
     END IF;
   ELSIF _ident = 'LS' THEN
     SELECT adtx_lieftxt, adtx_lieftxt_rtf INTO txt, txtrtf FROM artdokutxt WHERE adtx_ak_nr = _ak_nr AND adtx_spr_key = UPPER(intlang);
     IF NOT FOUND OR txt IS NULL THEN -- in abweichender Sprache nix gefunden, dann Artikeltext
        SELECT ak_lieftxt, ak_lieftxt_rtf INTO txt, txtrtf FROM art WHERE ak_nr = _ak_nr;
     END IF;
   ELSIF _ident = 'RE' THEN
     SELECT adtx_faktxt, adtx_faktxt_rtf INTO txt, txtrtf FROM artdokutxt WHERE adtx_ak_nr = _ak_nr AND adtx_spr_key = UPPER(intlang);
     IF NOT FOUND OR txt IS NULL THEN  -- in abweichender Sprache nix gefunden, dann Artikeltext
        SELECT ak_faktxt, ak_faktxt_rtf INTO txt, txtrtf FROM art WHERE ak_nr = _ak_nr;
     END IF;
   ELSIF _ident = 'EK' THEN
     SELECT adtx_bestxt, adtx_bestxt_rtf INTO txt, txtrtf FROM artdokutxt WHERE adtx_ak_nr = _ak_nr AND adtx_spr_key = UPPER(intlang);
     IF NOT FOUND OR txt IS NULL THEN -- in abweichender Sprache nix gefunden, dann Artikeltext
        SELECT ak_bestxt, ak_bestxt_rtf INTO txt, txtrtf FROM art WHERE ak_nr = _ak_nr;
     END IF;
   ELSE
     SELECT '', '' INTO txt, txtrtf;
   END IF;
   RETURN;
 END $$ LANGUAGE plpgsql STABLE;
--

 -- Mengencode zur Mengen-ID
CREATE OR REPLACE FUNCTION TArtikel.me__mec__by__mid(mid INTEGER) RETURNS INTEGER AS $$
  SELECT m_mgcode FROM artmgc WHERE m_id = mid;
  $$ LANGUAGE sql STABLE STRICT;
--
CREATE OR REPLACE FUNCTION TArtikel.ac__from__hoffmannshop_art(IN _ak_nr VARCHAR) RETURNS VARCHAR AS $$
  BEGIN
   IF (TSystem.Settings__Get('KUNDE')='KREYENBERG') THEN
      IF     (_ak_nr BETWEEN '010000%' AND '019999%') THEN RETURN 'PM1001';  --(Kalibrierservice)
      ELSEIF (_ak_nr BETWEEN '020000%' AND '089999%') THEN RETURN 'KT1030';  --(Werkstattbedarf, Schmiermittel)
      ELSEIF (_ak_nr BETWEEN '090000%' AND '099999%') THEN RETURN 'KT1020';  --(persönliche Schutzausrüstung und Bekleidung)
      ELSEIF (_ak_nr BETWEEN '111000%' AND '299999%') THEN RETURN 'WKZ1001'; --(Bohr- oder Zerspanwerkzeuge)
      ELSEIF (_ak_nr BETWEEN '300000%' AND '389999%') THEN RETURN 'WKZ1005'; --(Spanntechnik)
      ELSEIF (_ak_nr BETWEEN '410000%' AND '499999%') THEN RETURN 'PM1001';  --(Prüfmittel und Messgeräte)
      ELSEIF (_ak_nr BETWEEN '410000%' AND '499999%') THEN RETURN 'PM1000';  --(Prüfmittel und Messgeräte)
      ELSEIF (_ak_nr BETWEEN '510000%' AND '599999%') THEN RETURN 'KT1030';  --(Schleif- und Trenntechnik)
      ELSEIF (_ak_nr BETWEEN '610000%' AND '899999%') THEN RETURN 'WKZ1010'; --(Handwerkzeuge, Schraubenschlüssel, etc.)
      ELSEIF (_ak_nr BETWEEN '910000%' AND '999999%') THEN RETURN 'MA1002';  --(Werkstattausrüstung)
      ELSE                                                 RETURN 'S';
      END IF;
   ELSE
      RETURN null;
   END IF;
  END $$ LANGUAGE plpgsql;

--- #16871
CREATE OR REPLACE FUNCTION tartikel.artikelaustausch__db_link(
    IN ak_nr_old varchar,
    IN ak_nr_new varchar
    )
    RETURNS void
    AS $$
    BEGIN
      PERFORM * FROM dblink('host=localhost port=' || inet_server_port()::integer || ' dbname=' || current_database() || ' user=syncro password=syncro',
                            'SELECT tartikel.artikelaustausch('|| quote_literal(ak_nr_old) ||', ' || quote_literal(ak_nr_new) || ')'
                            ) AS sub (result varchar);
      RETURN;
    END $$ LANGUAGE plpgsql;
--

--- #20025, Verlinkung von Werkzeugen (oder z.B.: ESL (Electronic-Shelf-Label)) an andere Tabellen
CREATE TABLE werkzeug_link
  (
    wzl_id              serial PRIMARY KEY,
    wzl_wz_id           integer NOT null REFERENCES werkzeug (wz_id) ON UPDATE NO ACTION ON DELETE CASCADE, -- ID des Werkzeug/ESL
    wzl_table           regclass NOT null,                                                                  -- Zieltabelle auf die verlinkt werden soll
    wzl_pkey            varchar   -- Zieldatensatz
  );
--- Indizes
CREATE UNIQUE INDEX werkzeug_link__wzl__wz_id__table_ukey ON werkzeug_link (wzl_wz_id, wzl_table); -- jedes Werkzeug/ESL darf nur einmal pro Tabelle verknüpft werden
---